<?php

use Illuminate\Foundation\Inspiring;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Schedule;

Artisan::command('inspire', function () {
    $this->comment(Inspiring::quote());
})->purpose('Display an inspiring quote')->hourly();

// Schedule campaign auto-start (runs every minute)
Schedule::command('campaigns:start-scheduled')
    ->everyMinute()
    ->withoutOverlapping()
    ->runInBackground();

// Schedule autoresponder processing (runs every minute)
Schedule::command('autoresponders:process-due')
    ->everyMinute()
    ->withoutOverlapping()
    ->runInBackground();

Schedule::command('automations:process-due')
    ->everyMinute()
    ->withoutOverlapping()
    ->runInBackground();

// Schedule bounce processing (runs every 5 minutes)
Schedule::command('email:process-bounces --all')
    ->everyFiveMinutes()
    ->withoutOverlapping()
    ->runInBackground();

// Schedule reply processing (runs every 5 minutes)
Schedule::command('email:process-replies')
    ->everyFiveMinutes()
    ->withoutOverlapping()
    ->runInBackground();

// Schedule update check (runs every 10 minutes)
Schedule::command('updates:check --force')
    ->cron('0 */6 * * *')
    ->withoutOverlapping()
    ->runInBackground();

// Schedule update install (runs every 10 minutes)
Schedule::command('updates:install-if-available')
    ->everyTenMinutes()
    ->withoutOverlapping()
    ->runInBackground();

Schedule::command('subscriber-imports:process')
    ->everyMinute()
    ->withoutOverlapping()
    ->runInBackground();

Artisan::command('admin:create {email} {--first=Admin} {--last=User} {--password=} {--verify}', function () {
    $email = (string) $this->argument('email');
    $firstName = (string) $this->option('first');
    $lastName = (string) $this->option('last');
    $password = (string) ($this->option('password') ?: '');
    $verify = (bool) $this->option('verify');

    if ($password === '') {
        $password = $this->secret('Password');
    }

    if ($password === '' || strlen($password) < 8) {
        $this->error('Password is required and must be at least 8 characters.');
        return 1;
    }

    $adminGroup = \App\Models\UserGroup::firstOrCreate(
        ['name' => 'admin'],
        [
            'description' => 'System administrators',
            'permissions' => ['*'],
            'is_system' => true,
        ]
    );

    $user = \App\Models\User::withTrashed()->where('email', $email)->first();
    if ($user) {
        if ($user->trashed()) {
            $user->restore();
        }

        $user->update([
            'first_name' => $firstName ?: $user->first_name,
            'last_name' => $lastName ?: $user->last_name,
            'password' => \Illuminate\Support\Facades\Hash::make($password),
            'status' => 'active',
        ]);
    } else {
        $user = \App\Models\User::create([
            'first_name' => $firstName,
            'last_name' => $lastName,
            'email' => $email,
            'password' => \Illuminate\Support\Facades\Hash::make($password),
            'timezone' => 'UTC',
            'language' => 'en',
            'status' => 'active',
        ]);
    }

    $user->userGroups()->syncWithoutDetaching([$adminGroup->id]);

    if ($verify && $user->email_verified_at === null) {
        $user->forceFill(['email_verified_at' => now()])->save();
    }

    $this->info("Admin user ready: {$user->email}");
    $this->line("User ID: {$user->id}");
    $this->line("Admin group ID: {$adminGroup->id}");
    return 0;
})->purpose('Create/update an admin user and attach them to the admin user group');

