@extends('layouts.admin')

@section('title', 'Settings')
@section('page-title', 'Settings')

@section('content')
<div class="space-y-6">
    @if(session('success'))
        <div class="rounded-md bg-green-50 dark:bg-green-900/30 border border-green-200 dark:border-green-800 p-4">
            <div class="flex">
                <div class="flex-shrink-0">
                    <svg class="h-5 w-5 text-green-400" viewBox="0 0 20 20" fill="currentColor">
                        <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.857-9.809a.75.75 0 00-1.214-.882l-3.483 4.79-1.88-1.88a.75.75 0 10-1.06 1.061l2.5 2.5a.75.75 0 001.137-.089l4-5.5z" clip-rule="evenodd" />
                    </svg>
                </div>
                <div class="ml-3">
                    <p class="text-sm font-medium text-green-800 dark:text-green-200">{{ session('success') }}</p>
                </div>
            </div>
        </div>
    @endif

    @if(session('error'))
        <div class="rounded-md bg-red-50 dark:bg-red-900/30 border border-red-200 dark:border-red-800 p-4">
            <div class="flex">
                <div class="flex-shrink-0">
                    <svg class="h-5 w-5 text-red-400" viewBox="0 0 20 20" fill="currentColor">
                        <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.28 7.22a.75.75 0 00-1.06 1.06L8.94 10l-1.72 1.72a.75.75 0 101.06 1.06L10 11.06l1.72 1.72a.75.75 0 101.06-1.06L11.06 10l1.72-1.72a.75.75 0 00-1.06-1.06L10 8.94 8.28 7.22z" clip-rule="evenodd" />
                    </svg>
                </div>
                <div class="ml-3">
                    <p class="text-sm font-medium text-red-800 dark:text-red-200">{{ session('error') }}</p>
                </div>
            </div>
        </div>
    @endif
    <!-- Category Tabs -->
    @if($categories && count($categories) > 0)
        <div class="border-b border-gray-200 dark:border-gray-700 overflow-x-auto">
            <nav class="-mb-px flex min-w-max space-x-6 sm:space-x-8 px-2 sm:px-0" aria-label="Tabs">
                @php
                    $tabUpdateAvailable = false;
                    if (is_array($updateStatus ?? null)) {
                        $tabUpdateAvailable = (bool) ($updateStatus['update_available'] ?? false);
                    }
                @endphp
                @foreach($categories as $cat)
                    <a
                        href="{{ route('admin.settings.index', ['category' => $cat]) }}"
                        class="{{ $category === $cat ? '!border-primary-500 text-primary-600 dark:text-primary-400 border-b-2' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 dark:text-gray-400 dark:hover:text-gray-300' }} whitespace-nowrap shrink-0 py-3 sm:py-4 px-2 sm:px-1 font-medium text-sm"
                    >
                        <span class="inline-flex items-center gap-2">
                            <span>{{ ucfirst($cat) }}</span>
                            @if($cat === 'updates' && $tabUpdateAvailable)
                                <span class="inline-flex items-center justify-center px-2 py-0.5 rounded-full text-[10px] font-semibold bg-red-500 text-white">{{ __('Update') }}</span>
                            @endif
                        </span>
                    </a>
                @endforeach

                @admincan('admin.translations.access')
                    <a
                        href="{{ route('admin.translations.locales.index') }}"
                        class="{{ request()->routeIs('admin.translations.*') ? '!border-primary-500 text-primary-600 dark:text-primary-400 border-b-2' : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 dark:text-gray-400 dark:hover:text-gray-300' }} whitespace-nowrap shrink-0 py-3 sm:py-4 px-2 sm:px-1 font-medium text-sm"
                    >
                        Translations
                    </a>
                @endadmincan
            </nav>
        </div>
    @endif

    <!-- Settings Form -->
    <form method="POST" action="{{ route('admin.settings.update') }}" enctype="multipart/form-data">
        @csrf
        @method('POST')

        <input type="hidden" name="category" value="{{ $category }}">

        <x-card>
            @if($category === 'updates')
                @php
                    $installedVersion = null;
                    $installedVersion = trim((string) config('mailpurse.version', ''));
                    if ($installedVersion === '') {
                        $installedVersion = $settings['updates']->get('app_version') ?? null;
                        $installedVersion = is_string($installedVersion) ? trim((string) $installedVersion) : null;
                    }

                    $updateStatusPayload = is_array($updateStatus ?? null) ? $updateStatus : null;
                    $updateAvailableFlag = is_array($updateStatusPayload) ? (bool) ($updateStatusPayload['update_available'] ?? false) : false;
                    $updateStatusLatest = is_array($updateStatusPayload) && is_string($updateStatusPayload['latest_version'] ?? null) ? $updateStatusPayload['latest_version'] : null;
                    $updateStatusCheckedAt = is_array($updateStatusPayload) && is_string($updateStatusPayload['checked_at'] ?? null) ? $updateStatusPayload['checked_at'] : null;

                    $installTargetVersion = is_string($updateStatusLatest) && trim((string) $updateStatusLatest) !== ''
                        ? (string) $updateStatusLatest
                        : null;

                    $latestVersion = null;
                    $productData = null;
                    if (is_array($updateProduct ?? null)) {
                        $payload = $updateProduct['data'] ?? null;
                        if (is_array($payload)) {
                            if (is_array($payload['data'] ?? null)) {
                                $productData = $payload['data'];
                            } elseif (is_array($payload['product'] ?? null)) {
                                $productData = $payload['product'];
                            } else {
                                $productData = $payload;
                            }
                        }
                    }

                    if (is_array($productData)) {
                        $latestVersion = $productData['latest_version']
                            ?? ($productData['latestVersion'] ?? ($productData['latest'] ?? null));

                        if (!is_string($latestVersion) || trim((string) $latestVersion) === '') {
                            $latestVersion = $productData['version'] ?? null;
                        }
                    }

                    if (!is_string($latestVersion) || trim((string) $latestVersion) === '') {
                        $changelogData = is_array($updateChangelogs ?? null) ? ($updateChangelogs['data'] ?? null) : null;
                        if (is_array($changelogData)) {
                            $versionMap = null;
                            if (is_array($changelogData['releases'] ?? null)) {
                                $versionMap = $changelogData['releases'];
                            } elseif (is_array($changelogData['changelog'] ?? null)) {
                                $versionMap = $changelogData['changelog'];
                            }

                            if (is_array($versionMap) && !empty($versionMap)) {
                                $versions = array_keys($versionMap);
                                usort($versions, function ($a, $b) {
                                    return version_compare((string) $b, (string) $a);
                                });
                                $latestVersion = $versions[0] ?? null;
                            }
                        }
                    }

                    if (!$updateAvailableFlag && is_string($latestVersion) && is_string($installedVersion) && trim($latestVersion) !== '' && trim($installedVersion) !== '') {
                        $updateAvailableFlag = version_compare((string) $latestVersion, (string) $installedVersion, '>');
                    }

                    if (!is_string($installTargetVersion) || trim((string) $installTargetVersion) === '') {
                        $installTargetVersion = is_string($latestVersion) && trim((string) $latestVersion) !== ''
                            ? (string) $latestVersion
                            : null;
                    }
                @endphp

                <div class="space-y-6">
                    @if($updateAvailableFlag)
                        <div class="p-4 bg-blue-50 border border-blue-200 text-blue-800 rounded-lg dark:bg-blue-900/50 dark:border-blue-800 dark:text-blue-200">
                            <p class="font-semibold">Update available</p>
                            <p class="mt-1 text-sm">Installed: {{ is_string($installedVersion) && trim($installedVersion) !== '' ? $installedVersion : '—' }} | Latest: {{ is_string($updateStatusLatest) && trim($updateStatusLatest) !== '' ? $updateStatusLatest : (is_string($latestVersion) ? $latestVersion : '—') }}</p>
                            @if(is_string($updateStatusCheckedAt) && trim($updateStatusCheckedAt) !== '')
                                <p class="mt-1 text-xs">Last checked: {{ $updateStatusCheckedAt }}</p>
                            @endif
                        </div>
                    @endif
                    <div class="flex items-start justify-between gap-4">
                        <div>
                            <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100">Updates</h3>
                            <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Check the latest available version from your update server and generate a download link.</p>
                        </div>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div class="p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                            <p class="text-sm text-gray-500 dark:text-gray-400">Installed Version</p>
                            <p class="mt-1 text-lg font-semibold text-gray-900 dark:text-gray-100">{{ is_string($installedVersion) && trim($installedVersion) !== '' ? $installedVersion : '—' }}</p>
                        </div>
                        <div class="p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                            <p class="text-sm text-gray-500 dark:text-gray-400">Latest Version</p>
                            <p class="mt-1 text-lg font-semibold text-gray-900 dark:text-gray-100">
                                {{ is_string($latestVersion) && trim((string) $latestVersion) !== '' ? $latestVersion : '—' }}
                            </p>
                            @if(is_array($updateProduct ?? null) && !($updateProduct['success'] ?? false))
                                <p class="mt-2 text-sm text-red-600 dark:text-red-400">{{ $updateProduct['message'] ?? 'Unable to fetch update info.' }}</p>
                            @endif
                        </div>
                    </div>

                    <div class="p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                        <div class="flex items-start justify-between gap-4">
                            <div>
                                <p class="text-sm font-medium text-gray-900 dark:text-gray-100">Install Update</p>
                                <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Install the latest version in the background. The site will go into maintenance mode during installation.</p>
                            </div>
                            @admincan('admin.settings.edit')
                                <x-button
                                    type="submit"
                                    variant="primary"
                                    size="sm"
                                    formaction="{{ route('admin.settings.install-update') }}"
                                    formmethod="POST"
                                    :disabled="!$updateAvailableFlag"
                                    onclick="return confirm('Are you sure you want to install the update? The site will go into maintenance mode during the update.')"
                                >Install Update</x-button>
                            @endadmincan
                        </div>

                        @if(is_string($installTargetVersion ?? null) && trim((string) $installTargetVersion) !== '')
                            <input type="hidden" name="target_version" value="{{ $installTargetVersion }}">
                        @endif

                        @php
                            $installState = is_array($updateInstallState ?? null) ? $updateInstallState : null;
                            $installInProgress = is_array($installState) ? (bool) ($installState['in_progress'] ?? false) : false;
                            $installStatus = is_array($installState) && is_string($installState['status'] ?? null) ? $installState['status'] : null;
                            $installMessage = is_array($installState) && is_string($installState['message'] ?? null) ? $installState['message'] : null;
                            $installVersion = is_array($installState) && is_string($installState['version'] ?? null) ? $installState['version'] : null;
                        @endphp

                        @if($installInProgress || (is_string($installStatus) && trim((string) $installStatus) !== '') || (is_string($installMessage) && trim((string) $installMessage) !== ''))
                            <div class="mt-4 p-3 rounded-md border border-gray-200 dark:border-gray-700">
                                <p class="text-sm font-medium text-gray-900 dark:text-gray-100">Installer status</p>
                                <p class="mt-1 text-sm text-gray-600 dark:text-gray-300">
                                    {{ is_string($installStatus) ? ucfirst($installStatus) : '—' }}
                                    @if(is_string($installVersion) && trim($installVersion) !== '')
                                        ({{ $installVersion }})
                                    @endif
                                </p>
                                @if(is_string($installMessage) && trim($installMessage) !== '')
                                    <p class="mt-1 text-xs text-gray-500 dark:text-gray-400">{{ $installMessage }}</p>
                                @endif
                            </div>
                        @endif

                        @if(is_string($updateLastFailureReason ?? null) && trim((string) $updateLastFailureReason) !== '')
                            <div class="mt-4 p-3 rounded-md border border-red-200 bg-red-50 text-red-800 dark:border-red-800 dark:bg-red-900/30 dark:text-red-200">
                                <p class="text-sm font-semibold">Last update failed</p>
                                <p class="mt-1 text-sm">
                                    @if(is_string($updateLastFailureVersion ?? null) && trim((string) $updateLastFailureVersion) !== '')
                                        Version: {{ $updateLastFailureVersion }}
                                    @else
                                        Version: —
                                    @endif
                                    @if(is_string($updateLastFailureAt ?? null) && trim((string) $updateLastFailureAt) !== '')
                                        | Date: {{ $updateLastFailureAt }}
                                    @endif
                                </p>
                                <p class="mt-1 text-xs">{{ $updateLastFailureReason }}</p>
                            </div>
                        @endif

                        @if(is_string($updateDownloadUrl ?? null) && trim((string) $updateDownloadUrl) !== '')
                            <div class="mt-4">
                                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300">Download URL</label>
                                <div class="mt-2 flex flex-col sm:flex-row sm:items-center gap-2">
                                    <input
                                        type="text"
                                        readonly
                                        value="{{ $updateDownloadUrl }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                    <x-button href="{{ $updateDownloadUrl }}" variant="secondary" size="sm" target="_blank" rel="noopener noreferrer">Open</x-button>
                                </div>
                                <p class="mt-2 text-sm text-gray-500 dark:text-gray-400">This link may expire depending on your update server configuration.</p>
                            </div>
                        @endif
                    </div>

                    <div class="pt-2">
                        <h4 class="text-sm font-semibold text-gray-900 dark:text-gray-100">Configuration</h4>
                        
                        <div class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-6">

                            @php
                                $licenseCheckPayload = is_array($updateLicenseCheck ?? null) ? ($updateLicenseCheck['data'] ?? null) : null;
                                $licenseCheckTopStatus = is_array($licenseCheckPayload) && is_string($licenseCheckPayload['status'] ?? null) ? $licenseCheckPayload['status'] : null;
                                $licenseCheckTopMessage = is_array($licenseCheckPayload) && is_string($licenseCheckPayload['message'] ?? null) ? $licenseCheckPayload['message'] : null;

                                $licenseCheckRequestError = null;
                                if (is_array($updateLicenseCheck ?? null) && !($updateLicenseCheck['success'] ?? false)) {
                                    $licenseCheckRequestError = is_string($updateLicenseCheck['message'] ?? null) ? $updateLicenseCheck['message'] : null;
                                }

                                $licenseObject = null;
                                if (is_array($licenseCheckPayload) && is_array($licenseCheckPayload['data'] ?? null) && is_array($licenseCheckPayload['data']['license'] ?? null)) {
                                    $licenseObject = $licenseCheckPayload['data']['license'];
                                }

                                $savedLicense = null;
                                try {
                                    $savedLicense = \App\Models\Setting::get('update_license_key');
                                } catch (\Throwable $e) {
                                    $savedLicense = null;
                                }

                                $maskedLicense = null;
                                if (is_string($savedLicense) && trim($savedLicense) !== '') {
                                    $clean = trim((string) $savedLicense);
                                    if (preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $clean)) {
                                        $maskedLicense = substr($clean, 0, 8) . '-XXXX-XXXX-XXXX-' . substr($clean, -12);
                                    } else {
                                        $maskedLicense = substr($clean, 0, 8) . '…' . substr($clean, -6);
                                    }
                                }
                            @endphp

                            <div class="md:col-span-2">
                                <div class="p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                                    <div class="flex items-start justify-between gap-4">
                                        <div>
                                            <p class="text-sm font-medium text-gray-900 dark:text-gray-100">License</p>
                                            <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Status and support information from the license server.</p>
                                        </div>
                                        @admincan('admin.settings.edit')
                                            <div class="shrink-0">
                                                <x-button
                                                    type="button"
                                                    variant="danger"
                                                    size="sm"
                                                    onclick="if(confirm('Deactivate this license for the current domain?')) { deactivateLicense(); }"
                                                >Deactivate</x-button>
                                            </div>
                                        @endadmincan
                                    </div>

                                    @if(is_string($licenseCheckRequestError) && trim($licenseCheckRequestError) !== '')
                                        <p class="mt-3 text-sm text-red-600 dark:text-red-400">{{ $licenseCheckRequestError }}</p>
                                    @endif

                                    <div class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <p class="text-sm text-gray-500 dark:text-gray-400">Status</p>
                                            <p class="mt-1 text-sm font-semibold text-gray-900 dark:text-gray-100">{{ is_string($licenseObject['status'] ?? null) ? $licenseObject['status'] : ($licenseCheckTopStatus ?: '—') }}</p>
                                            @if(is_string($licenseCheckTopMessage) && trim($licenseCheckTopMessage) !== '')
                                                <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">{{ $licenseCheckTopMessage }}</p>
                                            @endif
                                        </div>

                                        <div>
                                            <p class="text-sm text-gray-500 dark:text-gray-400">License</p>
                                            <p class="mt-1 text-sm font-semibold text-gray-900 dark:text-gray-100">{{ $maskedLicense ?: '—' }}</p>
                                        </div>

                                        <div>
                                            <p class="text-sm text-gray-500 dark:text-gray-400">Support Expiry</p>
                                            <p class="mt-1 text-sm font-semibold text-gray-900 dark:text-gray-100">{{ is_string($licenseObject['supported_until'] ?? null) ? $licenseObject['supported_until'] : '—' }}</p>
                                        </div>

                                        <div>
                                            <p class="text-sm text-gray-500 dark:text-gray-400">License Type</p>
                                            <p class="mt-1 text-sm font-semibold text-gray-900 dark:text-gray-100">{{ is_string($licenseObject['license_type'] ?? null) ? $licenseObject['license_type'] : '—' }}</p>
                                        </div>

                                        <div>
                                            <p class="text-sm text-gray-500 dark:text-gray-400">Purchase Date</p>
                                            <p class="mt-1 text-sm font-semibold text-gray-900 dark:text-gray-100">{{ is_string($licenseObject['purchase_date'] ?? null) ? $licenseObject['purchase_date'] : '—' }}</p>
                                        </div>

                                        <div>
                                            <p class="text-sm text-gray-500 dark:text-gray-400">Envato Username</p>
                                            <p class="mt-1 text-sm font-semibold text-gray-900 dark:text-gray-100">{{ is_string($licenseObject['envato_username'] ?? null) ? $licenseObject['envato_username'] : '—' }}</p>
                                        </div>

                                        <div>
                                            <p class="text-sm text-gray-500 dark:text-gray-400">Domains</p>
                                            <p class="mt-1 text-sm font-semibold text-gray-900 dark:text-gray-100">
                                                {{ is_numeric($licenseObject['active_domains'] ?? null) ? (int) $licenseObject['active_domains'] : '—' }}
                                                /
                                                {{ is_numeric($licenseObject['max_domains'] ?? null) ? (int) $licenseObject['max_domains'] : '—' }}
                                            </p>
                                        </div>

                                        <div>
                                            <p class="text-sm text-gray-500 dark:text-gray-400">Normalized Domain</p>
                                            <p class="mt-1 text-sm font-semibold text-gray-900 dark:text-gray-100">{{ is_string($licenseObject['normalized_domain'] ?? null) ? $licenseObject['normalized_domain'] : '—' }}</p>
                                        </div>
                                    </div>

                                    @if(is_array($licenseObject) && is_array($licenseObject['domains'] ?? null) && !empty($licenseObject['domains']))
                                        <div class="mt-4">
                                            <p class="text-sm font-medium text-gray-900 dark:text-gray-100">Registered Domains</p>
                                            <div class="mt-2 overflow-x-auto">
                                                <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                                    <thead class="bg-gray-50 dark:bg-gray-800">
                                                        <tr>
                                                            <th scope="col" class="px-4 py-2 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Domain</th>
                                                            <th scope="col" class="px-4 py-2 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Status</th>
                                                        </tr>
                                                    </thead>
                                                    <tbody class="bg-white dark:bg-gray-900 divide-y divide-gray-200 dark:divide-gray-700">
                                                        @foreach($licenseObject['domains'] as $d => $st)
                                                            <tr>
                                                                <td class="px-4 py-2 text-sm text-gray-900 dark:text-gray-100">{{ is_string($d) ? $d : '' }}</td>
                                                                <td class="px-4 py-2 text-sm text-gray-700 dark:text-gray-300">{{ is_string($st) ? $st : '' }}</td>
                                                            </tr>
                                                        @endforeach
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    @endif
                                </div>
                            </div>


                            <div class="md:col-span-2">
                                <label for="setting_update_license_key" class="block text-sm font-medium text-gray-700 dark:text-gray-300">License Key</label>
                                <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Enter your purchase code / license key and click Activate.</p>
                                <div class="mt-2 flex gap-2">
                                    <input
                                        type="text"
                                        name="update_license_key"
                                        id="setting_update_license_key"
                                        value="{{ old('update_license_key') }}"
                                        autocomplete="new-password"
                                        placeholder="xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                    <button
                                        type="button"
                                        onclick="activateLicense()"
                                        class="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-primary-600 hover:bg-primary-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-primary-500 whitespace-nowrap"
                                    >
                                        Activate
                                    </button>
                                </div>
                                @error('update_license_key')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>

                            <script>
                                function activateLicense() {
                                    var licenseKey = document.getElementById('setting_update_license_key').value;
                                    var form = document.createElement('form');
                                    form.method = 'POST';
                                    form.action = '{{ route('admin.settings.license-activate') }}';
                                    
                                    var csrf = document.createElement('input');
                                    csrf.type = 'hidden';
                                    csrf.name = '_token';
                                    csrf.value = '{{ csrf_token() }}';
                                    form.appendChild(csrf);
                                    
                                    var input = document.createElement('input');
                                    input.type = 'hidden';
                                    input.name = 'update_license_key';
                                    input.value = licenseKey;
                                    form.appendChild(input);
                                    
                                    document.body.appendChild(form);
                                    form.submit();
                                }

                                function deactivateLicense() {
                                    var form = document.createElement('form');
                                    form.method = 'POST';
                                    form.action = '{{ route('admin.settings.license-deactivate') }}';
                                    
                                    var csrf = document.createElement('input');
                                    csrf.type = 'hidden';
                                    csrf.name = '_token';
                                    csrf.value = '{{ csrf_token() }}';
                                    form.appendChild(csrf);
                                    
                                    document.body.appendChild(form);
                                    form.submit();
                                }
                            </script>
                        </div>
                    </div>
                </div>

            @elseif($category === 'changelogs')
                @php
                    $items = [];
                    $changelogMap = [];
                    $releasesMap = [];
                    $sortedVersions = [];
                    if (is_array($updateChangelogs ?? null)) {
                        $data = $updateChangelogs['data'] ?? null;

                        if (is_array($data) && is_array($data['changelog'] ?? null)) {
                            $changelogMap = $data['changelog'];
                            $releasesMap = is_array($data['releases'] ?? null) ? $data['releases'] : [];
                        } elseif (is_array($data) && is_array($data['changelogs'] ?? null)) {
                            $items = $data['changelogs'];
                        } elseif (is_array($data) && is_array($data['data'] ?? null)) {
                            $items = $data['data'];
                        } elseif (is_array($data)) {
                            $items = $data;
                        }
                    }

                    if (is_array($changelogMap) && !empty($changelogMap)) {
                        $sortedVersions = array_keys($changelogMap);
                        usort($sortedVersions, function ($a, $b) {
                            return version_compare((string) $b, (string) $a);
                        });
                    }
                @endphp

                <div class="space-y-6">
                    <div class="flex items-start justify-between gap-4">
                        <div>
                            <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100">Changelogs</h3>
                            <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Release notes fetched from your update server.</p>
                        </div>
                    </div>

                    @if(is_array($updateChangelogs ?? null) && !($updateChangelogs['success'] ?? false))
                        <div class="p-4 bg-red-50 border border-red-200 text-red-800 rounded-lg dark:bg-red-900/50 dark:border-red-800 dark:text-red-200">
                            {{ $updateChangelogs['message'] ?? 'Unable to fetch changelogs.' }}
                        </div>
                    @endif

                    @if(!empty($sortedVersions))
                        <div class="space-y-4">
                            @foreach($sortedVersions as $version)
                                @php
                                    $release = is_array($releasesMap[$version] ?? null) ? $releasesMap[$version] : [];
                                    $releaseDate = is_string($release['release_date'] ?? null) ? $release['release_date'] : null;

                                    $releaseChangelog = is_array($release['changelog'] ?? null) ? $release['changelog'] : null;
                                    $entry = is_array($releaseChangelog) ? $releaseChangelog : (is_array($changelogMap[$version] ?? null) ? $changelogMap[$version] : []);
                                @endphp

                                <div class="p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                                    <div class="flex items-center justify-between gap-4">
                                        <p class="font-medium text-gray-900 dark:text-gray-100">{{ $version }}</p>
                                        <p class="text-sm text-gray-500 dark:text-gray-400">{{ $releaseDate ?: '' }}</p>
                                    </div>

                                    <div class="mt-3 space-y-3">
                                        @foreach($entry as $type => $messages)
                                            @php
                                                $title = is_string($type) ? ucfirst(str_replace('_', ' ', $type)) : 'Changes';
                                                $lines = [];

                                                $messageList = [];
                                                if (is_string($messages) && trim($messages) !== '') {
                                                    $messageList = [$messages];
                                                } elseif (is_array($messages)) {
                                                    $messageList = $messages;
                                                }

                                                if (!empty($messageList)) {
                                                    foreach ($messageList as $msg) {
                                                        if (!is_string($msg)) {
                                                            continue;
                                                        }

                                                        $parts = array_map('trim', explode(',', $msg));
                                                        foreach ($parts as $part) {
                                                            if ($part !== '') {
                                                                $lines[] = $part;
                                                            }
                                                        }
                                                    }
                                                }
                                            @endphp

                                            @if(!empty($lines))
                                                <div>
                                                    <p class="text-sm font-semibold text-gray-900 dark:text-gray-100">{{ $title }}</p>
                                                    <ul class="mt-2 list-disc pl-5 space-y-1 text-sm text-gray-700 dark:text-gray-300">
                                                        @foreach($lines as $line)
                                                            <li>{{ $line }}</li>
                                                        @endforeach
                                                    </ul>
                                                </div>
                                            @endif
                                        @endforeach
                                    </div>
                                </div>
                            @endforeach
                        </div>
                    @elseif(!empty($items))
                        <div class="space-y-4">
                            @foreach($items as $item)
                                @php
                                    $version = is_array($item) && is_string($item['version'] ?? null) ? $item['version'] : null;
                                    $date = is_array($item) && is_string($item['date'] ?? null) ? $item['date'] : null;
                                    $content = is_array($item) && is_string($item['content'] ?? null) ? $item['content'] : null;
                                @endphp

                                <div class="p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                                    <div class="flex items-center justify-between gap-4">
                                        <p class="font-medium text-gray-900 dark:text-gray-100">{{ $version ?: 'Release' }}</p>
                                        <p class="text-sm text-gray-500 dark:text-gray-400">{{ $date ?: '' }}</p>
                                    </div>
                                    @if($content)
                                        <div class="mt-3 prose prose-sm max-w-none dark:prose-invert">
                                            {!! $content !!}
                                        </div>
                                    @endif
                                </div>
                            @endforeach
                        </div>
                    @else
                        <div class="text-center py-12 text-gray-500 dark:text-gray-400">
                            <p>No changelog entries available.</p>
                        </div>
                    @endif
                </div>

            @elseif($category === 'cron')
                @php
                    $settingsByKey = $settings->get($category, collect())->keyBy('key');
                    $cronEnabled = (bool) ($settingsByKey['cron_web_enabled']->value ?? false);
                    $cronToken = is_string($settingsByKey['cron_web_token']->value ?? null)
                        ? (string) $settingsByKey['cron_web_token']->value
                        : '';
                    $cronLastRunAt = is_string($settingsByKey['cron_last_run_at']->value ?? null)
                        ? (string) $settingsByKey['cron_last_run_at']->value
                        : '';
                    $cronUrl = route('cron.run', ['token' => $cronToken]);
                    $cronWgetCommand = 'wget -q -O - "' . $cronUrl . '" > /dev/null 2>&1';
                    $cronCurlCommand = 'curl -fsS "' . $cronUrl . '" > /dev/null 2>&1';
                    $cronRunOutput = session('cron_run_output');
                @endphp

                <div class="space-y-6">
                    <div>
                        <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100">Cron / Scheduler</h3>
                        <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Use this for cPanel cron jobs (or any external scheduler) to trigger Laravel's scheduler.</p>
                    </div>

                    <div class="p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                        <div class="flex items-start justify-between gap-4">
                            <div>
                                <p class="text-sm font-medium text-gray-900 dark:text-gray-100">Web Cron</p>
                                <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">When enabled, calls to the Web Cron URL will run <code>schedule:run</code>.</p>
                            </div>
                            <div class="shrink-0">
                                <input type="hidden" name="cron_web_enabled" value="0">
                                <label class="inline-flex items-center cursor-pointer">
                                    <input
                                        type="checkbox"
                                        name="cron_web_enabled"
                                        id="setting_cron_web_enabled"
                                        value="1"
                                        {{ $cronEnabled ? 'checked' : '' }}
                                        class="sr-only peer"
                                    >
                                    <div class="relative w-11 h-6 bg-gray-200 rounded-full peer-focus:outline-none peer-focus:ring-2 peer-focus:ring-primary-500 dark:bg-gray-700 peer-checked:bg-primary-600 after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border after:border-gray-300 after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:after:translate-x-full peer-checked:after:border-white"></div>
                                </label>
                            </div>
                        </div>
                        @error('cron_web_enabled')
                            <p class="mt-2 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                        @enderror
                    </div>

                    <div class="p-4 rounded-lg border border-gray-200 dark:border-gray-700 space-y-4">
                        <div>
                            <label class="block text-sm font-medium text-gray-700 dark:text-gray-300">Web Cron URL</label>
                            <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Add this URL in cPanel Cron Jobs (every minute). Keep the token secret.</p>
                            <div class="mt-2">
                                <input
                                    type="text"
                                    readonly
                                    value="{{ $cronUrl }}"
                                    class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                >
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300">cPanel command (wget)</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        readonly
                                        value="{{ $cronWgetCommand }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray-700 dark:text-gray-300">cPanel command (curl)</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        readonly
                                        value="{{ $cronCurlCommand }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                            </div>
                        </div>

                        <div class="flex flex-wrap items-center gap-2">
                            @admincan('admin.settings.edit')
                                <x-button
                                    type="submit"
                                    variant="secondary"
                                    size="sm"
                                    formaction="{{ route('admin.settings.cron.regenerate-token') }}"
                                    formmethod="POST"
                                    onclick="return confirm('Regenerate token? You must update your cron job URL in cPanel after this.')"
                                >Regenerate Token</x-button>
                                <x-button
                                    type="submit"
                                    variant="primary"
                                    size="sm"
                                    formaction="{{ route('admin.settings.cron.run-now') }}"
                                    formmethod="POST"
                                >Run Now</x-button>
                            @endadmincan
                        </div>
                    </div>

                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div class="p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                            <p class="text-sm text-gray-500 dark:text-gray-400">Last run</p>
                            <p class="mt-1 text-sm font-semibold text-gray-900 dark:text-gray-100">{{ $cronLastRunAt !== '' ? $cronLastRunAt : '—' }}</p>
                        </div>
                        <div class="p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                            <p class="text-sm text-gray-500 dark:text-gray-400">Status</p>
                            <p class="mt-1 text-sm font-semibold {{ $cronEnabled ? 'text-green-600 dark:text-green-400' : 'text-gray-700 dark:text-gray-300' }}">{{ $cronEnabled ? 'Enabled' : 'Disabled' }}</p>
                        </div>
                    </div>

                    @if(is_string($cronRunOutput) && trim((string) $cronRunOutput) !== '')
                        <div class="p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                            <p class="text-sm font-medium text-gray-900 dark:text-gray-100">Last run output</p>
                            <pre class="mt-2 text-xs whitespace-pre-wrap text-gray-700 dark:text-gray-300">{{ $cronRunOutput }}</pre>
                        </div>
                    @endif
                </div>

            @elseif($category === 'storage')
                @php
                    $settingsByKey = $settings->get($category, collect())->keyBy('key');
                    $getVal = function (string $key, $default = null) {
                        try {
                            return \App\Models\Setting::get($key, $default);
                        } catch (\Throwable $e) {
                            return $default;
                        }
                    };

                    $s3Configured = is_string($getVal('s3_key')) && trim((string) $getVal('s3_key')) !== ''
                        && is_string($getVal('s3_secret')) && trim((string) $getVal('s3_secret')) !== ''
                        && is_string($getVal('s3_region')) && trim((string) $getVal('s3_region')) !== ''
                        && is_string($getVal('s3_bucket')) && trim((string) $getVal('s3_bucket')) !== '';

                    $wasabiConfigured = is_string($getVal('wasabi_key')) && trim((string) $getVal('wasabi_key')) !== ''
                        && is_string($getVal('wasabi_secret')) && trim((string) $getVal('wasabi_secret')) !== ''
                        && is_string($getVal('wasabi_region')) && trim((string) $getVal('wasabi_region')) !== ''
                        && is_string($getVal('wasabi_bucket')) && trim((string) $getVal('wasabi_bucket')) !== ''
                        && is_string($getVal('wasabi_endpoint')) && trim((string) $getVal('wasabi_endpoint')) !== '';

                    $gcsConfigured = is_string($getVal('gcs_project_id')) && trim((string) $getVal('gcs_project_id')) !== ''
                        && is_string($getVal('gcs_bucket')) && trim((string) $getVal('gcs_bucket')) !== ''
                        && is_string($getVal('gcs_key_file')) && trim((string) $getVal('gcs_key_file')) !== '';
                @endphp

                <div class="space-y-8">
                    <div>
                        <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100">Providers</h3>
                        <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Enable/disable providers. The app will use your default storage from General settings, and fall back to Local if the provider is disabled or not configured.</p>

                        <div class="mt-4 overflow-x-auto">
                            <table class="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                <thead class="bg-gray-50 dark:bg-gray-800">
                                    <tr>
                                        <th scope="col" class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Provider</th>
                                        <th scope="col" class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Active</th>
                                        <th scope="col" class="px-4 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">Configured</th>
                                    </tr>
                                </thead>
                                <tbody class="bg-white dark:bg-gray-900 divide-y divide-gray-200 dark:divide-gray-700">
                                    <tr>
                                        <td class="px-4 py-3 text-sm text-gray-900 dark:text-gray-100">Local</td>
                                        <td class="px-4 py-3">
                                            <label class="inline-flex items-center">
                                                <input
                                                    type="checkbox"
                                                    name="storage_local_enabled"
                                                    value="1"
                                                    {{ ($settingsByKey['storage_local_enabled']->value ?? false) ? 'checked' : '' }}
                                                    class="rounded border-gray-300 text-primary-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:border-gray-600 dark:bg-gray-700"
                                                >
                                                <span class="ml-2 text-sm text-gray-700 dark:text-gray-300">Enabled</span>
                                            </label>
                                            @error('storage_local_enabled')
                                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                            @enderror
                                        </td>
                                        <td class="px-4 py-3 text-sm text-gray-700 dark:text-gray-300">Yes</td>
                                    </tr>

                                    <tr>
                                        <td class="px-4 py-3 text-sm text-gray-900 dark:text-gray-100">Amazon S3</td>
                                        <td class="px-4 py-3">
                                            <label class="inline-flex items-center">
                                                <input
                                                    type="checkbox"
                                                    name="storage_s3_enabled"
                                                    value="1"
                                                    {{ ($settingsByKey['storage_s3_enabled']->value ?? false) ? 'checked' : '' }}
                                                    class="rounded border-gray-300 text-primary-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:border-gray-600 dark:bg-gray-700"
                                                >
                                                <span class="ml-2 text-sm text-gray-700 dark:text-gray-300">Enabled</span>
                                            </label>
                                            @error('storage_s3_enabled')
                                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                            @enderror
                                        </td>
                                        <td class="px-4 py-3 text-sm {{ $s3Configured ? 'text-green-600 dark:text-green-400' : 'text-gray-500 dark:text-gray-400' }}">
                                            {{ $s3Configured ? 'Yes' : 'No' }}
                                        </td>
                                    </tr>

                                    <tr>
                                        <td class="px-4 py-3 text-sm text-gray-900 dark:text-gray-100">Wasabi</td>
                                        <td class="px-4 py-3">
                                            <label class="inline-flex items-center">
                                                <input
                                                    type="checkbox"
                                                    name="storage_wasabi_enabled"
                                                    value="1"
                                                    {{ ($settingsByKey['storage_wasabi_enabled']->value ?? false) ? 'checked' : '' }}
                                                    class="rounded border-gray-300 text-primary-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:border-gray-600 dark:bg-gray-700"
                                                >
                                                <span class="ml-2 text-sm text-gray-700 dark:text-gray-300">Enabled</span>
                                            </label>
                                            @error('storage_wasabi_enabled')
                                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                            @enderror
                                        </td>
                                        <td class="px-4 py-3 text-sm {{ $wasabiConfigured ? 'text-green-600 dark:text-green-400' : 'text-gray-500 dark:text-gray-400' }}">
                                            {{ $wasabiConfigured ? 'Yes' : 'No' }}
                                        </td>
                                    </tr>

                                    <tr>
                                        <td class="px-4 py-3 text-sm text-gray-900 dark:text-gray-100">Google Cloud Storage</td>
                                        <td class="px-4 py-3">
                                            <label class="inline-flex items-center">
                                                <input
                                                    type="checkbox"
                                                    name="storage_gcs_enabled"
                                                    value="1"
                                                    {{ ($settingsByKey['storage_gcs_enabled']->value ?? false) ? 'checked' : '' }}
                                                    class="rounded border-gray-300 text-primary-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:border-gray-600 dark:bg-gray-700"
                                                >
                                                <span class="ml-2 text-sm text-gray-700 dark:text-gray-300">Enabled</span>
                                            </label>
                                            @error('storage_gcs_enabled')
                                                <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                            @enderror
                                        </td>
                                        <td class="px-4 py-3 text-sm {{ $gcsConfigured ? 'text-green-600 dark:text-green-400' : 'text-gray-500 dark:text-gray-400' }}">
                                            {{ $gcsConfigured ? 'Yes' : 'No' }}
                                        </td>
                                    </tr>
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <div>
                        <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100">Common</h3>
                        <div class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label for="setting_storage_public_root" class="block text-sm font-medium text-gray-700 dark:text-gray-300">Storage Public Root</label>
                                <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Root folder/prefix to store public assets under (optional).</p>
                                <div class="mt-2">
                                    @php
                                        $storagePublicRootOptions = [
                                            '' => 'None',
                                            'public' => 'public',
                                            'uploads' => 'uploads',
                                            'assets' => 'assets',
                                            'media' => 'media',
                                        ];

                                        $storagePublicRootValue = $getVal('storage_public_root', 'public');
                                        $storagePublicRootValue = is_string($storagePublicRootValue) ? trim($storagePublicRootValue) : '';
                                        $storagePublicRootIsCustom = !array_key_exists($storagePublicRootValue, $storagePublicRootOptions);
                                    @endphp
                                    <select
                                        name="storage_public_root"
                                        id="setting_storage_public_root"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                        @if($storagePublicRootIsCustom)
                                            <option value="{{ $storagePublicRootValue }}" selected>
                                                {{ $storagePublicRootValue === '' ? 'None' : $storagePublicRootValue }}
                                            </option>
                                        @endif
                                        @foreach($storagePublicRootOptions as $optValue => $optLabel)
                                            <option value="{{ $optValue }}" {{ (string) $storagePublicRootValue === (string) $optValue ? 'selected' : '' }}>
                                                {{ $optLabel }}
                                            </option>
                                        @endforeach
                                    </select>
                                </div>
                                @error('storage_public_root')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div>
                        <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100">Amazon S3 Settings</h3>
                        <div class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label for="setting_s3_key" class="block text-sm font-medium text-gray-700 dark:text-gray-300">S3 Key</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="s3_key"
                                        id="setting_s3_key"
                                        value="{{ $getVal('s3_key') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('s3_key')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_s3_secret" class="block text-sm font-medium text-gray-700 dark:text-gray-300">S3 Secret</label>
                                <div class="mt-2">
                                    <input
                                        type="password"
                                        name="s3_secret"
                                        id="setting_s3_secret"
                                        value=""
                                        autocomplete="new-password"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('s3_secret')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_s3_region" class="block text-sm font-medium text-gray-700 dark:text-gray-300">S3 Region</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="s3_region"
                                        id="setting_s3_region"
                                        value="{{ $getVal('s3_region') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('s3_region')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_s3_bucket" class="block text-sm font-medium text-gray-700 dark:text-gray-300">S3 Bucket</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="s3_bucket"
                                        id="setting_s3_bucket"
                                        value="{{ $getVal('s3_bucket') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('s3_bucket')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_s3_endpoint" class="block text-sm font-medium text-gray-700 dark:text-gray-300">S3 Endpoint</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="s3_endpoint"
                                        id="setting_s3_endpoint"
                                        value="{{ $getVal('s3_endpoint') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('s3_endpoint')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_s3_url" class="block text-sm font-medium text-gray-700 dark:text-gray-300">S3 URL</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="s3_url"
                                        id="setting_s3_url"
                                        value="{{ $getVal('s3_url') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('s3_url')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div>
                        <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100">Wasabi Settings</h3>
                        <div class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label for="setting_wasabi_key" class="block text-sm font-medium text-gray-700 dark:text-gray-300">Wasabi Key</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="wasabi_key"
                                        id="setting_wasabi_key"
                                        value="{{ $getVal('wasabi_key') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('wasabi_key')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_wasabi_secret" class="block text-sm font-medium text-gray-700 dark:text-gray-300">Wasabi Secret</label>
                                <div class="mt-2">
                                    <input
                                        type="password"
                                        name="wasabi_secret"
                                        id="setting_wasabi_secret"
                                        value=""
                                        autocomplete="new-password"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('wasabi_secret')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_wasabi_region" class="block text-sm font-medium text-gray-700 dark:text-gray-300">Wasabi Region</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="wasabi_region"
                                        id="setting_wasabi_region"
                                        value="{{ $getVal('wasabi_region', 'us-east-1') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('wasabi_region')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_wasabi_bucket" class="block text-sm font-medium text-gray-700 dark:text-gray-300">Wasabi Bucket</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="wasabi_bucket"
                                        id="setting_wasabi_bucket"
                                        value="{{ $getVal('wasabi_bucket') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('wasabi_bucket')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_wasabi_endpoint" class="block text-sm font-medium text-gray-700 dark:text-gray-300">Wasabi Endpoint</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="wasabi_endpoint"
                                        id="setting_wasabi_endpoint"
                                        value="{{ $getVal('wasabi_endpoint') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('wasabi_endpoint')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_wasabi_url" class="block text-sm font-medium text-gray-700 dark:text-gray-300">Wasabi URL</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="wasabi_url"
                                        id="setting_wasabi_url"
                                        value="{{ $getVal('wasabi_url') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('wasabi_url')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                    </div>

                    <div>
                        <h3 class="text-base font-semibold text-gray-900 dark:text-gray-100">Google Cloud Storage Settings</h3>
                        <div class="mt-4 grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label for="setting_gcs_project_id" class="block text-sm font-medium text-gray-700 dark:text-gray-300">GCS Project ID</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="gcs_project_id"
                                        id="setting_gcs_project_id"
                                        value="{{ $getVal('gcs_project_id') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('gcs_project_id')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_gcs_bucket" class="block text-sm font-medium text-gray-700 dark:text-gray-300">GCS Bucket</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="gcs_bucket"
                                        id="setting_gcs_bucket"
                                        value="{{ $getVal('gcs_bucket') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('gcs_bucket')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_gcs_key_file" class="block text-sm font-medium text-gray-700 dark:text-gray-300">GCS Key File</label>
                                <p class="mt-1 text-sm text-gray-500 dark:text-gray-400">Absolute path to service account JSON file on the server.</p>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="gcs_key_file"
                                        id="setting_gcs_key_file"
                                        value="{{ $getVal('gcs_key_file') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('gcs_key_file')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_gcs_path_prefix" class="block text-sm font-medium text-gray-700 dark:text-gray-300">GCS Path Prefix</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="gcs_path_prefix"
                                        id="setting_gcs_path_prefix"
                                        value="{{ $getVal('gcs_path_prefix') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('gcs_path_prefix')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                            <div>
                                <label for="setting_gcs_url" class="block text-sm font-medium text-gray-700 dark:text-gray-300">GCS URL</label>
                                <div class="mt-2">
                                    <input
                                        type="text"
                                        name="gcs_url"
                                        id="setting_gcs_url"
                                        value="{{ $getVal('gcs_url') }}"
                                        class="block w-full rounded-md border-gray-300 dark:border-gray-600 shadow-sm focus:border-primary-500 focus:ring-primary-500 dark:bg-gray-700 dark:text-gray-100 sm:text-sm"
                                    >
                                </div>
                                @error('gcs_url')
                                    <p class="mt-1 text-sm text-red-600 dark:text-red-400">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>
                    </div>
                </div>
            @elseif($category === 'ai' && $settings->has($category) && $settings[$category]->count() > 0)
                @php
                    $aiSettings = $settings[$category];
                    $openaiSettings = $aiSettings->filter(fn ($s) => \Illuminate\Support\Str::startsWith((string) $s->key, 'openai_'));
                    $geminiSettings = $aiSettings->filter(fn ($s) => \Illuminate\Support\Str::startsWith((string) $s->key, 'gemini_'));
                    $claudeSettings = $aiSettings->filter(fn ($s) => \Illuminate\Support\Str::startsWith((string) $s->key, 'claude_'));

                    $otherAiSettings = $aiSettings->reject(function ($s) {
                        return \Illuminate\Support\Str::startsWith((string) $s->key, 'openai_')
                            || \Illuminate\Support\Str::startsWith((string) $s->key, 'gemini_')
                            || \Illuminate\Support\Str::startsWith((string) $s->key, 'claude_');
                    });

                    $groups = [
                        ['title' => 'OpenAI / GPT', 'items' => $openaiSettings],
                        ['title' => 'Gemini', 'items' => $geminiSettings],
                        ['title' => 'Claude', 'items' => $claudeSettings],
                    ];
                @endphp

                <div class="space-y-4">
                    @foreach($groups as $group)
                        @php
                            $items = $group['items'] ?? collect();
                        @endphp

                        @if($items->count() > 0)
                            <details class="rounded-lg border border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800" open>
                                <summary class="cursor-pointer select-none px-4 py-3 text-sm font-semibold text-gray-900 dark:text-gray-100">
                                    {{ $group['title'] ?? '' }}
                                </summary>
                                <div class="px-4 pb-4 pt-1 grid grid-cols-1 md:grid-cols-2 gap-6">
                                    @foreach($items as $setting)
                                        @include('admin.settings._setting_field', ['setting' => $setting])
                                    @endforeach
                                </div>
                            </details>
                        @endif
                    @endforeach

                    @if($otherAiSettings->count() > 0)
                        <details class="rounded-lg border border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
                            <summary class="cursor-pointer select-none px-4 py-3 text-sm font-semibold text-gray-900 dark:text-gray-100">
                                {{ __('Other') }}
                            </summary>
                            <div class="px-4 pb-4 pt-1 grid grid-cols-1 md:grid-cols-2 gap-6">
                                @foreach($otherAiSettings as $setting)
                                    @include('admin.settings._setting_field', ['setting' => $setting])
                                @endforeach
                            </div>
                        </details>
                    @endif
                </div>

            @elseif($category === 'navigation' && $settings->has($category) && $settings[$category]->count() > 0)
                @php
                    $navigationSettings = $settings[$category]
                        ->reject(fn ($s) => in_array((string) ($s->key ?? ''), ['nav_show_home_dropdown', 'nav_show_home_1', 'nav_show_home_2', 'nav_show_home_3', 'nav_show_home_4'], true));
                @endphp
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    @foreach($navigationSettings as $setting)
                        @include('admin.settings._setting_field', ['setting' => $setting])
                    @endforeach
                </div>

            @elseif($settings->has($category) && $settings[$category]->count() > 0)
                <div class="space-y-6">
                    @foreach($settings[$category] as $setting)
                        @include('admin.settings._setting_field', ['setting' => $setting])
                    @endforeach
                </div>
            @else
                <div class="text-center py-12 text-gray-500 dark:text-gray-400">
                    <p>No settings found for this category.</p>
                    <p class="mt-2 text-sm">Settings will appear here once they are created.</p>
                </div>
            @endif

            <!-- Form Actions -->
            @if($settings->has($category) && $settings[$category]->count() > 0)
                <div class="flex items-center justify-end gap-4 pt-6 mt-6 border-t border-gray-200 dark:border-gray-700">
                    @admincan('admin.settings.edit')
                        <x-button type="submit" variant="primary">Save Settings</x-button>
                    @endadmincan
                </div>
            @endif
        </x-card>
    </form>
</div>
@endsection

