import './bootstrap';
import Alpine from 'alpinejs';
import collapse from '@alpinejs/collapse';
import '@hotwired/turbo';

Alpine.plugin(collapse);
window.Alpine = Alpine;

if (window.Turbo && window.Turbo.session) {
    window.Turbo.session.drive = true;
}

if (window.Turbo && typeof window.Turbo.setProgressBarDelay === 'function') {
    window.Turbo.setProgressBarDelay(0);
}

Alpine.data('templateImportModal', (opts = {}) => ({
    importOpen: false,
    canImport: !!opts.canImport,
    canAi: !!opts.canAi,
    tab: opts.initialTab || 'templates',
    loading: false,
    previewLoading: false,
    importing: false,
    error: '',
    gallery: [],
    categories: [],
    activeCategory: 'all',
    selected: null,
    preview: null,
    previewModalOpen: false,
    previewModalItem: null,
    previewModalLoading: false,
    previewModalData: null,
    ai: {
        provider: 'chatgpt',
        model: '',
        prompt: '',
        loading: false,
        error: '',
    },

    init() {
        if (this.$watch) {
            this.$watch('ai.provider', () => {
                const allowed = this.aiModelsForProvider();
                if (this.ai.model && !allowed.includes(this.ai.model)) {
                    this.ai.model = '';
                }
            });
        }
    },

    aiModelsForProvider() {
        const provider = (this.ai && typeof this.ai.provider === 'string') ? this.ai.provider : 'chatgpt';

        if (provider === 'gemini') {
            return ['gemini-2.0-flash', 'gemini-2.0-pro', 'gemini-1.5-flash-latest', 'gemini-1.5-pro-latest', 'gemini-1.0-pro-latest'];
        }

        if (provider === 'claude') {
            return ['claude-3-5-sonnet-20241022', 'claude-3-5-haiku-20241022', 'claude-3-opus-20240229'];
        }

        return ['gpt-5', 'gpt-5-mini', 'gpt-5.2', 'gpt-5-nano', 'gpt-4.1'];
    },

    open() {
        this.importOpen = true;
        this.error = '';
        this.selected = null;
        this.preview = null;
        this.activeCategory = 'all';
        this.previewModalOpen = false;
        this.previewModalItem = null;
        this.previewModalData = null;
        this.ai.error = '';

        if (this.tab !== 'ai' && !this.canImport) {
            this.tab = 'ai';
        }

        if (this.tab !== 'ai' && this.canImport) {
            this.loadGallery();
        }
    },

    close() {
        this.importOpen = false;
        this.error = '';
    },

    setTab(tab) {
        if (tab === 'templates' && !this.canImport) {
            this.tab = 'ai';
            return;
        }
        if (tab === 'ai' && !this.canAi) {
            return;
        }

        this.tab = tab;
        this.error = '';
        this.selected = null;
        this.preview = null;
        this.activeCategory = 'all';
        this.ai.error = '';

        if (this.tab !== 'ai' && this.canImport) {
            this.loadGallery();
        }
    },

    setCategory(categoryId) {
        this.activeCategory = categoryId;
    },

    get filteredGallery() {
        if (this.activeCategory === 'all') {
            return this.gallery;
        }
        return this.gallery.filter(item => item.category === this.activeCategory);
    },

    get visibleCategories() {
        const usedCategories = new Set(this.gallery.map(item => item.category || 'other'));
        return this.categories.filter(cat => cat.id === 'all' || usedCategories.has(cat.id));
    },

    loadGallery() {
        if (!this.canImport || !opts.galleryUrl) {
            this.gallery = [];
            return;
        }

        this.loading = true;
        this.error = '';

        const tabParam = 'templates';
        const builderParam = opts.builder || 'grapesjs';
        const url = (opts.galleryUrl || '') + '?tab=' + encodeURIComponent(tabParam) + '&builder=' + encodeURIComponent(builderParam);
        fetch(url, { headers: { 'Accept': 'application/json' } })
            .then((r) => r.ok ? r.json() : Promise.reject(r))
            .then((data) => {
                this.gallery = Array.isArray(data.templates) ? data.templates : [];
                this.categories = Array.isArray(data.categories) ? data.categories : [];
            })
            .catch(() => {
                this.error = 'Failed to load templates.';
            })
            .finally(() => {
                this.loading = false;
            });
    },

    openPreviewModal(item) {
        if (!item) return;
        this.previewModalItem = item;
        this.previewModalOpen = true;
        this.previewModalLoading = true;
        this.previewModalData = null;

        const url = (item && typeof item.content_url === 'string' && item.content_url)
            ? item.content_url
            : ((opts.contentUrlBase || '') + '/' + item.id + '/content');

        fetch(url, { headers: { 'Accept': 'application/json' } })
            .then((r) => r.ok ? r.json() : Promise.reject(r))
            .then((data) => {
                this.previewModalData = data;
            })
            .catch(() => {
                this.error = 'Failed to load template preview.';
            })
            .finally(() => {
                this.previewModalLoading = false;
            });
    },

    closePreviewModal() {
        this.previewModalOpen = false;
        this.previewModalItem = null;
        this.previewModalData = null;
    },

    insertFromPreviewModal() {
        if (!this.previewModalData) return;
        this.preview = this.previewModalData;
        this.closePreviewModal();
        this.doImport();
    },

    insertTemplate(item) {
        if (!item) return;
        this.previewLoading = true;
        this.error = '';

        const url = (item && typeof item.content_url === 'string' && item.content_url)
            ? item.content_url
            : ((opts.contentUrlBase || '') + '/' + item.id + '/content');

        fetch(url, { headers: { 'Accept': 'application/json' } })
            .then((r) => r.ok ? r.json() : Promise.reject(r))
            .then((data) => {
                this.preview = data;
                this.doImport();
            })
            .catch(() => {
                this.error = 'Failed to load template.';
            })
            .finally(() => {
                this.previewLoading = false;
            });
    },

    select(item) {
        if (!this.canImport || !opts.contentUrlBase) {
            return;
        }

        this.selected = item;
        this.previewLoading = true;
        this.error = '';

        const url = (item && typeof item.content_url === 'string' && item.content_url)
            ? item.content_url
            : ((opts.contentUrlBase || '') + '/' + item.id + '/content');

        fetch(url, { headers: { 'Accept': 'application/json' } })
            .then((r) => r.ok ? r.json() : Promise.reject(r))
            .then((data) => {
                this.preview = data;
            })
            .catch(() => {
                this.error = 'Failed to load template preview.';
            })
            .finally(() => {
                this.previewLoading = false;
            });
    },

    doImport() {
        if (!this.preview) {
            return;
        }
        this.importing = true;
        this.error = '';

        const builder = this.preview.builder || null;

        const originInput = document.querySelector('input[name="settings[origin]"]');
        if (originInput) {
            originInput.value = this.tab === 'ai' ? 'ai' : '';
        }

        if (opts.builder === 'unlayer' && builder === 'unlayer' && window.unlayer && this.preview.builder_data) {
            const waitForUnlayer = (timeoutMs = 8000) => new Promise((resolve, reject) => {
                const start = Date.now();
                const tick = () => {
                    if (window.__mailpurseUnlayerReady === true) {
                        resolve();
                        return;
                    }
                    if (Date.now() - start >= timeoutMs) {
                        reject(new Error('Unlayer editor is not ready yet.'));
                        return;
                    }
                    setTimeout(tick, 75);
                };
                tick();
            });

            waitForUnlayer()
                .then(() => {
                    const rawDesign = this.preview.builder_data;
                    const unwrapped = (window.Alpine && typeof window.Alpine.raw === 'function')
                        ? window.Alpine.raw(rawDesign)
                        : rawDesign;

                    let design = unwrapped;
                    if (typeof design === 'string') {
                        const trimmed = design.trim();
                        if (trimmed) {
                            try {
                                design = JSON.parse(trimmed);
                            } catch (e) {
                                throw new Error('Template design is not valid JSON.');
                            }
                        }
                    }

                    if (!design || typeof design !== 'object') {
                        throw new Error('Template design is invalid.');
                    }

                    if (!design.body && (design.rows || design.values)) {
                        design = { counters: {}, body: design };
                    }

                    if (!design.body || typeof design.body !== 'object') {
                        throw new Error('Template design is missing body.');
                    }

                    try {
                        if (typeof structuredClone === 'function') {
                            design = structuredClone(design);
                        } else {
                            design = JSON.parse(JSON.stringify(design));
                        }
                    } catch (e) {
                        try {
                            design = JSON.parse(JSON.stringify(design));
                        } catch (e2) {
                            const msg = (e2 && e2.message) ? e2.message : ((e && e.message) ? e.message : 'Unknown error');
                            throw new Error('Template design could not be cloned: ' + msg);
                        }
                    }

                    window.unlayer.loadDesign(design);
                    this.importOpen = false;
                })
                .catch((e) => {
                    const msg = (e && e.message) ? e.message : null;
                    this.error = msg ? ('Failed to import this template: ' + msg) : 'Failed to import this template.';
                })
                .finally(() => {
                    this.importing = false;
                });
            return;
        }

        if (opts.builder === 'grapesjs' && builder === 'grapesjs' && window.__mailpurseGrapesEditor) {
            try {
                const editor = window.__mailpurseGrapesEditor;
                const builderData = this.preview.builder_data || null;
                const components = builderData && builderData.components ? builderData.components : null;
                const styles = builderData && builderData.styles ? builderData.styles : null;
                const html = this.preview.html_content || null;

                if (components) {
                    editor.setComponents(components);
                } else if (html) {
                    editor.setComponents(html);
                }
                if (styles) {
                    editor.setStyle(styles);
                }

                this.importOpen = false;
            } catch (e) {
                this.error = 'Failed to import this template.';
            } finally {
                this.importing = false;
            }
            return;
        }

        this.importing = false;
        this.error = 'This template is not compatible with the current builder.';
    },

    importSelected() {
        this.doImport();
    },

    aiGenerate() {
        this.ai.loading = true;
        this.ai.error = '';
        this.error = '';

        fetch(opts.aiUrl, {
            method: 'POST',
            headers: {
                'Accept': 'application/json',
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': opts.csrfToken,
            },
            body: JSON.stringify({
                provider: this.ai.provider,
                model: this.ai.model || null,
                prompt: this.ai.prompt,
                builder: opts.builder || 'grapesjs',
            }),
        })
            .then(async (r) => {
                if (!r.ok) {
                    const payload = await r.json().catch(() => null);
                    const message = payload && payload.message ? payload.message : 'Failed to generate template.';
                    throw new Error(message);
                }
                return r.json();
            })
            .then((data) => {
                this.preview = data;
            })
            .catch((e) => {
                this.ai.error = (e && e.message) ? e.message : 'Failed to generate template.';
            })
            .finally(() => {
                this.ai.loading = false;
            });
    },
}));

Alpine.data('headerSearch', (config = {}) => ({
    query: config.initialQuery || '',
    open: false,
    loading: false,
    items: [],
    activeIndex: -1,
    debounceMs: config.debounceMs || 200,
    minChars: config.minChars || 2,
    suggestUrl: config.suggestUrl || '',
    searchUrl: config.searchUrl || '',
    variant: config.variant || 'admin',
    _debounceTimer: null,
    _abortController: null,

    get hasItems() {
        return Array.isArray(this.items) && this.items.length > 0;
    },

    get searchUrlWithQuery() {
        if (!this.searchUrl) {
            return '#';
        }
        const q = (this.query || '').trim();
        return q ? `${this.searchUrl}?q=${encodeURIComponent(q)}` : this.searchUrl;
    },

    get dropdownBgClass() {
        return this.variant === 'customer' ? 'bg-admin-sidebar border-gray-500' : 'bg-admin-sidebar border-gray-500';
    },

    get itemHoverClass() {
        return this.variant === 'customer' ? 'hover:bg-white/5' : 'hover:bg-white/5';
    },

    reset() {
        this.items = [];
        this.activeIndex = -1;
        this.loading = false;
    },

    close() {
        this.open = false;
        this.activeIndex = -1;
    },

    onFocus() {
        const q = (this.query || '').trim();
        if (q.length >= this.minChars) {
            if (this.hasItems) {
                this.open = true;
            } else {
                this.fetchSuggestions();
            }
        }
    },

    onInput() {
        clearTimeout(this._debounceTimer);
        this._debounceTimer = setTimeout(() => {
            this.fetchSuggestions();
        }, this.debounceMs);
    },

    async fetchSuggestions() {
        const q = (this.query || '').trim();

        if (!this.suggestUrl || q.length < this.minChars) {
            this.reset();
            this.open = false;
            return;
        }

        if (this._abortController) {
            this._abortController.abort();
        }

        this._abortController = new AbortController();
        this.loading = true;
        this.open = true;

        try {
            const res = await window.axios.get(this.suggestUrl, {
                params: { q },
                signal: this._abortController.signal,
            });
            this.items = res?.data?.items || [];
            this.activeIndex = this.items.length ? 0 : -1;
        } catch (e) {
            if (e?.name !== 'CanceledError' && e?.code !== 'ERR_CANCELED') {
                this.items = [];
                this.activeIndex = -1;
            }
        } finally {
            this.loading = false;
        }
    },

    onKeydown(e) {
        if (!this.open) {
            return;
        }

        if (e.key === 'Escape') {
            e.preventDefault();
            this.close();
            return;
        }

        if (e.key === 'ArrowDown') {
            e.preventDefault();
            if (!this.hasItems) {
                return;
            }
            this.activeIndex = Math.min(this.items.length - 1, this.activeIndex + 1);
            return;
        }

        if (e.key === 'ArrowUp') {
            e.preventDefault();
            if (!this.hasItems) {
                return;
            }
            this.activeIndex = Math.max(0, this.activeIndex - 1);
            return;
        }

        if (e.key === 'Enter') {
            if (this.activeIndex >= 0 && this.items[this.activeIndex]?.url) {
                e.preventDefault();
                window.location.href = this.items[this.activeIndex].url;
            }
        }
    },

    select(item) {
        if (item?.url) {
            window.location.href = item.url;
        }
    },
}));

Alpine.start();

function escapeHtml(unsafe) {
    if (unsafe === null || unsafe === undefined) {
        return '';
    }
    return String(unsafe)
        .replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#039;');
}

function getNotificationsUiVariant(feedUrl) {
    if (!feedUrl) {
        return 'admin';
    }
    return feedUrl.includes('/customer/') ? 'customer' : 'admin';
}

function getNotificationsUiTokens(variant) {
    if (variant === 'customer') {
        return {
            emptyTextClass: 'text-gray-500 dark:text-gray-400',
            itemTitleClass: 'text-gray-900 dark:text-gray-100 font-medium',
            itemMessageClass: 'mt-1 text-xs text-gray-600 dark:text-gray-300',
            itemTimeClass: 'mt-1 text-[11px] text-gray-400',
            itemReadBgClass: 'bg-white dark:bg-gray-800',
            itemUnreadBgClass: 'bg-primary-50/70 dark:bg-primary-900/20',
        };
    }

    return {
        emptyTextClass: 'text-admin-text-secondary',
        itemTitleClass: 'text-admin-text-primary font-medium',
        itemMessageClass: 'mt-1 text-xs text-admin-text-secondary',
        itemTimeClass: 'mt-1 text-[11px] text-admin-text-secondary/80',
        itemReadBgClass: 'bg-admin-sidebar',
        itemUnreadBgClass: 'bg-white/5',
    };
}

function renderNotificationsList(listEl, notifications, variant) {
    if (!listEl) {
        return;
    }

    const t = getNotificationsUiTokens(variant);

    if (!Array.isArray(notifications) || notifications.length === 0) {
        listEl.innerHTML = `
            <div class="px-4 py-6 text-center text-sm ${t.emptyTextClass}">
                No notifications yet.
            </div>
        `;
        return;
    }

    listEl.innerHTML = notifications
        .map((n) => {
            const title = escapeHtml(n.title || 'Notification');
            const message = n.message ? `<p class="${t.itemMessageClass}">${escapeHtml(n.message)}</p>` : '';
            const time = escapeHtml(n.created_at_human || '');
            const isRead = !!n.is_read;
            const bgClass = isRead ? t.itemReadBgClass : t.itemUnreadBgClass;

            const url = (n && typeof n === 'object' && n.data && typeof n.data === 'object' && typeof n.data.url === 'string')
                ? n.data.url
                : null;

            const wrapperTagOpen = url
                ? `<a href="${escapeHtml(url)}" class="block px-4 py-3 text-sm ${bgClass}">`
                : `<div class="px-4 py-3 text-sm ${bgClass}">`;
            const wrapperTagClose = url ? '</a>' : '</div>';

            return `
                ${wrapperTagOpen}
                    <p class="${t.itemTitleClass}">${title}</p>
                    ${message}
                    <p class="${t.itemTimeClass}">${time}</p>
                ${wrapperTagClose}
            `;
        })
        .join('');
}

function updateBadge(badgeEl, unreadCount) {
    if (!badgeEl) {
        return;
    }
    const count = Number(unreadCount) || 0;
    if (count <= 0) {
        badgeEl.remove();
        return;
    }
    badgeEl.textContent = count > 9 ? '9+' : String(count);
}

function ensureBadge(rootButtonEl, unreadCount) {
    const count = Number(unreadCount) || 0;
    const existing = rootButtonEl?.querySelector('[data-notifications-badge]') || null;
    if (count <= 0) {
        if (existing) {
            existing.remove();
        }
        return;
    }
    if (existing) {
        existing.textContent = count > 9 ? '9+' : String(count);
        return;
    }

    const span = document.createElement('span');
    span.setAttribute('data-notifications-badge', '');
    span.className = 'absolute -top-0.5 -right-0.5 inline-flex items-center justify-center px-1.5 py-0.5 rounded-full text-[10px] font-semibold bg-red-500 text-white';
    span.textContent = count > 9 ? '9+' : String(count);
    rootButtonEl.appendChild(span);
}

function updateUnreadLabel(labelEl, unreadCount) {
    if (!labelEl) {
        return;
    }
    const count = Number(unreadCount) || 0;
    labelEl.textContent = count > 0 ? `${count} unread` : 'Up to date';
}

async function refreshNotifications(rootButtonEl, containerEl) {
    const feedUrl = rootButtonEl?.getAttribute('data-notifications-feed-url');
    if (!feedUrl) {
        return;
    }

    const scope = containerEl || document;
    const listEl = scope.querySelector('[data-notifications-list]');
    const labelEl = scope.querySelector('[data-notifications-unread-label]');

    try {
        const res = await window.axios.get(feedUrl);
        const unreadCount = res?.data?.unread_count ?? 0;
        const notifications = res?.data?.notifications ?? [];

        ensureBadge(rootButtonEl, unreadCount);
        updateUnreadLabel(labelEl, unreadCount);

        const variant = getNotificationsUiVariant(feedUrl);
        renderNotificationsList(listEl, notifications, variant);
    } catch (e) {
        // Silent failure: avoid spamming console in production
    }
}

function setupNotificationWidget(rootButtonEl, allWidgets) {
    if (!rootButtonEl || !window.axios) {
        return;
    }

    const containerEl = rootButtonEl.parentElement;
    const markAllReadButton = (containerEl || document).querySelector('[data-notifications-mark-all-read]');
    const markAllReadUrl = rootButtonEl.getAttribute('data-notifications-mark-all-read-url');

    if (markAllReadButton && markAllReadUrl) {
        markAllReadButton.addEventListener('click', async (e) => {
            e.preventDefault();
            try {
                await window.axios.post(markAllReadUrl);
            } finally {
                (allWidgets || []).forEach((btn) => {
                    refreshNotifications(btn, btn.parentElement);
                });
            }
        });
    }

    // Refresh when the dropdown is opened
    rootButtonEl.addEventListener('click', () => {
        refreshNotifications(rootButtonEl, containerEl);
    });
}

function setupNotificationPolling() {
    const rootButtons = document.querySelectorAll('[data-notifications-root]');
    if (!rootButtons.length) {
        return;
    }

    const widgets = Array.from(rootButtons);
    widgets.forEach((btn) => setupNotificationWidget(btn, widgets));

    // Initial fetch + periodic refresh (single interval for all widgets)
    const refreshAll = () => {
        widgets.forEach((btn) => {
            refreshNotifications(btn, btn.parentElement);
        });
    };

    refreshAll();

    if (window.__mailpurseNotificationsIntervalId) {
        clearInterval(window.__mailpurseNotificationsIntervalId);
    }
    window.__mailpurseNotificationsIntervalId = setInterval(refreshAll, 15000);
}

function ensureUnlayerScriptLoaded() {
    if (window.unlayer && typeof window.unlayer.init === 'function') {
        return Promise.resolve();
    }

    if (window.__mailpurseUnlayerEmbedPromise) {
        return window.__mailpurseUnlayerEmbedPromise;
    }

    window.__mailpurseUnlayerEmbedPromise = new Promise((resolve, reject) => {
        try {
            const existing = document.querySelector('script[src="https://editor.unlayer.com/embed.js"]');
            if (existing) {
                const tick = (timeoutMs = 12000) => {
                    const started = Date.now();
                    const wait = () => {
                        if (window.unlayer && typeof window.unlayer.init === 'function') {
                            resolve();
                            return;
                        }
                        if (Date.now() - started >= timeoutMs) {
                            reject(new Error('Unlayer embed.js did not load in time.'));
                            return;
                        }
                        setTimeout(wait, 75);
                    };
                    wait();
                };
                tick();
                return;
            }

            const script = document.createElement('script');
            script.src = 'https://editor.unlayer.com/embed.js';
            script.async = true;
            script.onload = () => resolve();
            script.onerror = () => reject(new Error('Failed to load Unlayer embed.js.'));
            document.head.appendChild(script);
        } catch (e) {
            reject(e);
        }
    });

    return window.__mailpurseUnlayerEmbedPromise;
}

function setupUnlayerEditors() {
    const editorEls = document.querySelectorAll('[data-unlayer-editor]');
    if (!editorEls.length) {
        return;
    }

    ensureUnlayerScriptLoaded()
        .then(() => {
            editorEls.forEach((el) => {
                const id = el.getAttribute('id') || 'editor-container';
                const initKey = '__mailpurseUnlayerInitDone:' + id;

                if (window[initKey]) {
                    return;
                }

                const waitForVisibleAndInit = (timeoutMs = 12000) => {
                    const started = Date.now();
                    const tick = () => {
                        if (window[initKey]) {
                            return;
                        }

                        // Turbo can fire turbo:load before layout settles; avoid init on 0-height containers.
                        if (el.offsetParent === null || el.offsetHeight === 0) {
                            if (Date.now() - started >= timeoutMs) {
                                return;
                            }
                            setTimeout(tick, 75);
                            return;
                        }

                        const projectIdRaw = el.getAttribute('data-unlayer-project-id');
                        const projectId = projectIdRaw ? Number(projectIdRaw) : null;
                        const displayMode = el.getAttribute('data-unlayer-display-mode') || 'email';

                        const options = {
                            id,
                            displayMode,
                        };

                        if (projectId) {
                            options.projectId = projectId;
                        }

                        window[initKey] = true;
                        window.__mailpurseUnlayerReady = false;

                        window.unlayer.init(options);

                        window.unlayer.addEventListener('editor:ready', () => {
                            window.__mailpurseUnlayerReady = true;

                            const designScriptId = el.getAttribute('data-unlayer-design-script-id');
                            if (designScriptId) {
                                const designEl = document.getElementById(designScriptId);
                                const raw = designEl ? (designEl.textContent || '') : '';
                                if (raw && raw.trim()) {
                                    try {
                                        const design = JSON.parse(raw);
                                        if (design && typeof design === 'object') {
                                            window.unlayer.loadDesign(design);
                                        }
                                    } catch (e) {
                                        // ignore invalid design
                                    }
                                }
                            }
                        });
                    };
                    tick();
                };

                waitForVisibleAndInit();
            });
        })
        .catch(() => {
            // silent
        });
}

function setupUnlayerSaveBindings() {
    const form = document.getElementById('unlayer-form');
    const saveButton = document.getElementById('btn-save');

    if (!form || !saveButton) {
        return;
    }

    if (saveButton.dataset.unlayerBound === '1') {
        return;
    }
    saveButton.dataset.unlayerBound = '1';

    saveButton.addEventListener('click', () => {
        if (!window.unlayer || typeof window.unlayer.exportHtml !== 'function') {
            return;
        }

        window.unlayer.exportHtml((data) => {
            const htmlInput = document.getElementById('html_content');
            const plainInput = document.getElementById('plain_text_content');
            const dataInput = document.getElementById('grapesjs_data');

            if (htmlInput) {
                htmlInput.value = (data && data.html) ? data.html : '';
            }

            if (dataInput) {
                dataInput.value = JSON.stringify((data && data.design) ? data.design : null);
            }

            const plainText = String((data && data.html) ? data.html : '')
                .replace(/<style[\s\S]*?<\/style>/gi, ' ')
                .replace(/<script[\s\S]*?<\/script>/gi, ' ')
                .replace(/<[^>]+>/g, ' ')
                .replace(/\s+/g, ' ')
                .trim();

            if (plainInput) {
                plainInput.value = plainText;
            }

            form.submit();
        });
    });
}

function teardownUnlayerEditorsBeforeCache() {
    const editorEls = document.querySelectorAll('[data-unlayer-editor]');
    editorEls.forEach((el) => {
        const id = el.getAttribute('id') || 'editor-container';
        const initKey = '__mailpurseUnlayerInitDone:' + id;
        try {
            delete window[initKey];
        } catch (e) {
        }
        el.innerHTML = '';
    });

    const saveButton = document.getElementById('btn-save');
    if (saveButton && saveButton.dataset) {
        delete saveButton.dataset.unlayerBound;
    }

    window.__mailpurseUnlayerReady = false;
}

const runAppBoot = () => {
    setupNotificationPolling();
    setupUnlayerEditors();
    setupUnlayerSaveBindings();
};

if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', runAppBoot);
} else {
    runAppBoot();
}

document.addEventListener('turbo:before-cache', () => {
    if (window.Alpine && typeof window.Alpine.destroyTree === 'function') {
        window.Alpine.destroyTree(document.body);
    }

    teardownUnlayerEditorsBeforeCache();
});

document.addEventListener('turbo:load', () => {
    if (window.Alpine && typeof window.Alpine.initTree === 'function') {
        if (!document.documentElement.__x) {
            window.Alpine.initTree(document.documentElement);
        }
        window.Alpine.initTree(document.body);
    }
    runAppBoot();
});

