<?php

namespace App\Services;

use App\Models\TrackingDomain;
use App\Models\Customer;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;

class TrackingDomainService
{
    /**
     * Get paginated list of tracking domains for a customer.
     */
    public function getPaginated(Customer $customer, array $filters = [], int $perPage = 15): LengthAwarePaginator
    {
        $mustAddOwn = (bool) $customer->groupSetting('domains.tracking_domains.must_add', false);
        $canUseSystem = (bool) $customer->groupSetting('servers.permissions.can_use_system_servers', false);

        $query = TrackingDomain::query()
            ->when($mustAddOwn, function ($q) use ($customer) {
                $q->where('customer_id', $customer->id);
            }, function ($q) use ($customer, $canUseSystem) {
                $q->where(function ($sub) use ($customer, $canUseSystem) {
                    $sub->where('customer_id', $customer->id);
                    if ($canUseSystem) {
                        $sub->orWhereNull('customer_id');
                    }
                });
            });

        // Apply filters
        if (isset($filters['search'])) {
            $search = $filters['search'];
            $query->where(function ($q) use ($search) {
                $q->where('domain', 'like', "%{$search}%");
            });
        }

        if (isset($filters['status'])) {
            $query->where('status', $filters['status']);
        }

        return $query->latest()->paginate($perPage);
    }

    /**
     * Create a new tracking domain.
     */
    public function create(Customer $customer, array $data): TrackingDomain
    {
        $domain = strtolower(trim($data['domain']));

        $existing = TrackingDomain::withTrashed()
            ->where('domain', $domain)
            ->first();

        if ($existing !== null) {
            if ($existing->trashed() && (int) $existing->customer_id === (int) $customer->id) {
                $existing->restore();

                $existing->update([
                    'customer_id' => $customer->id,
                    'status' => 'pending',
                    'verification_token' => Str::random(32),
                    'dns_records' => $data['dns_records'] ?? [],
                    'notes' => $data['notes'] ?? null,
                    'verified_at' => null,
                    'verification_data' => null,
                ]);

                return $existing->fresh();
            }

            throw ValidationException::withMessages([
                'domain' => 'This domain has already been added.',
            ]);
        }

        return TrackingDomain::create([
            'customer_id' => $customer->id,
            'domain' => $domain,
            'status' => 'pending',
            'verification_token' => Str::random(32),
            'dns_records' => $data['dns_records'] ?? [],
            'notes' => $data['notes'] ?? null,
        ]);
    }

    /**
     * Update an existing tracking domain.
     */
    public function update(TrackingDomain $trackingDomain, array $data): TrackingDomain
    {
        $trackingDomain->update($data);
        return $trackingDomain->fresh();
    }

    /**
     * Delete a tracking domain.
     */
    public function delete(TrackingDomain $trackingDomain): bool
    {
        return $trackingDomain->delete();
    }

    /**
     * Verify a tracking domain.
     */
    public function verify(TrackingDomain $trackingDomain): bool
    {
        // In a real implementation, this would check DNS records
        $trackingDomain->update([
            'status' => 'verified',
            'verified_at' => now(),
        ]);

        return true;
    }
}

