<?php

namespace App\Services;

use App\Models\Template;
use App\Models\Customer;
use Illuminate\Pagination\LengthAwarePaginator;
use Illuminate\Support\Str;

class TemplateService
{
    /**
     * Get paginated list of templates for a customer.
     */
    public function getPaginated(Customer $customer, array $filters = [], int $perPage = 15): LengthAwarePaginator
    {
        $query = Template::where(function ($q) use ($customer) {
            $q->where('customer_id', $customer->id)
              ->orWhere(function ($subQ) {
                  $subQ->where('is_public', true)
                       ->where('is_system', false);
              });
        });

        // Apply filters
        if (isset($filters['search'])) {
            $search = $filters['search'];
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('description', 'like', "%{$search}%");
            });
        }

        if (isset($filters['type'])) {
            $query->where('type', $filters['type']);
        }

        return $query->latest()->paginate($perPage);
    }

    /**
     * Create a new template.
     */
    public function create(Customer $customer, array $data): Template
    {
        $settings = [];
        if (isset($data['settings']) && is_array($data['settings'])) {
            $settings = $data['settings'];
        }

        return Template::create([
            'customer_id' => $customer->id,
            'name' => $data['name'],
            'description' => $data['description'] ?? null,
            'type' => $data['type'] ?? 'email',
            'html_content' => $data['html_content'] ?? null,
            'plain_text_content' => $data['plain_text_content'] ?? null,
            'grapesjs_data' => $data['grapesjs_data'] ?? null,
            'settings' => $settings,
            'is_public' => $data['is_public'] ?? false,
        ]);
    }

    /**
     * Update an existing template.
     */
    public function update(Template $template, array $data): Template
    {
        $settings = is_array($template->settings) ? $template->settings : [];
        if (isset($data['settings']) && is_array($data['settings'])) {
            $settings = array_replace($settings, $data['settings']);
        }

        $template->update([
            'name' => $data['name'] ?? $template->name,
            'description' => $data['description'] ?? $template->description,
            'type' => $data['type'] ?? $template->type,
            'html_content' => $data['html_content'] ?? $template->html_content,
            'plain_text_content' => $data['plain_text_content'] ?? $template->plain_text_content,
            'grapesjs_data' => $data['grapesjs_data'] ?? $template->grapesjs_data,
            'settings' => $settings,
            'is_public' => $data['is_public'] ?? $template->is_public,
        ]);

        return $template->fresh();
    }

    /**
     * Delete a template.
     */
    public function delete(Template $template): bool
    {
        // Don't allow deletion of system templates
        if ($template->is_system) {
            throw new \Exception('Cannot delete system templates.');
        }

        return $template->delete();
    }

    /**
     * Duplicate a template.
     */
    public function duplicate(Template $template, Customer $customer): Template
    {
        $newTemplate = $template->replicate();
        $newTemplate->customer_id = $customer->id;
        $newTemplate->name = $template->name . ' (Copy)';
        $newTemplate->slug = null; // Will be auto-generated
        $newTemplate->is_public = false;
        $newTemplate->is_system = false;
        $newTemplate->usage_count = 0;
        $newTemplate->save();

        return $newTemplate;
    }
}

