<?php

namespace App\Services;

use App\Models\Campaign;
use App\Models\CampaignLog;
use App\Models\CampaignReply;
use App\Models\CampaignRecipient;
use App\Models\ListSubscriber;
use App\Models\ReplyServer;
use Carbon\Carbon;
use App\Services\AutomationTriggerService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;

class ReplyProcessorService
{
    public function processReplies(?ReplyServer $replyServer = null): int
    {
        if (!(bool) config('mailpurse.reply_tracking.enabled', false)) {
            return 0;
        }

        if (!function_exists('imap_open')) {
            Log::error('IMAP extension is not available. Please install php-imap extension.');
            throw new \Exception('IMAP extension is not available. Please install php-imap extension.');
        }

        $imap = $this->resolveImapSettings($replyServer);

        $hostname = (string) ($imap['hostname'] ?? '');
        $username = (string) ($imap['username'] ?? '');
        $password = (string) ($imap['password'] ?? '');
        $mailbox = (string) ($imap['mailbox'] ?? 'INBOX');

        $port = (int) ($imap['port'] ?? 993);
        $encryption = (string) ($imap['encryption'] ?? 'ssl');

        $deleteAfter = (bool) ($imap['delete_after_processing'] ?? false);
        $maxBatch = (int) ($imap['max_emails_per_batch'] ?? 50);
        if ($maxBatch <= 0) {
            $maxBatch = 50;
        }

        if ($hostname === '' || $username === '' || $password === '') {
            Log::warning('Reply tracking IMAP is enabled but missing connection settings.', [
                'reply_server_id' => $replyServer?->id,
            ]);
            return 0;
        }

        $connectionString = "{{$hostname}:{$port}";
        if ($encryption === 'ssl') {
            $connectionString .= '/ssl';
        } elseif ($encryption === 'tls') {
            $connectionString .= '/tls';
        }
        $connectionString .= '}';
        $connectionString .= $mailbox;

        $connection = @imap_open($connectionString, $username, $password);
        if (!$connection) {
            throw new \Exception('Failed to connect to reply inbox: ' . imap_last_error());
        }

        $messages = imap_search($connection, 'UNSEEN');
        if (!$messages) {
            imap_close($connection);
            return 0;
        }

        $messages = array_slice($messages, 0, $maxBatch);

        $processed = 0;
        foreach ($messages as $messageNumber) {
            try {
                $header = (string) imap_fetchheader($connection, $messageNumber);
                $body = (string) imap_body($connection, $messageNumber);
                $raw = $header . "\r\n\r\n" . $body;

                if ($this->isForwardedMessage($raw)) {
                    imap_setflag_full($connection, (string) $messageNumber, "\\Seen");
                    if ($deleteAfter) {
                        imap_delete($connection, $messageNumber);
                    }
                    continue;
                }

                $recipientUuid = $this->extractRecipientUuid($raw);
                if (!$recipientUuid) {
                    imap_setflag_full($connection, (string) $messageNumber, "\\Seen");
                    if ($deleteAfter) {
                        imap_delete($connection, $messageNumber);
                    }
                    continue;
                }

                $from = $this->extractHeaderValue($raw, 'From');
                $subject = $this->extractHeaderValue($raw, 'Subject');
                $messageId = $this->extractHeaderValue($raw, 'Message-ID');
                $date = $this->extractHeaderValue($raw, 'Date');
                $bodyText = $this->extractPlainBody($raw);

                $recipient = CampaignRecipient::query()->where('uuid', $recipientUuid)->first();
                if (!$recipient) {
                    imap_setflag_full($connection, (string) $messageNumber, "\\Seen");
                    if ($deleteAfter) {
                        imap_delete($connection, $messageNumber);
                    }
                    continue;
                }

                $recipient->loadMissing('campaign');
                $campaign = $recipient->campaign;

                $messageKey = $this->normalizeMessageId($messageId);

                $wasReplied = $recipient->isReplied();

                DB::transaction(function () use ($campaign, $recipient, $from, $subject, $messageId, $messageKey, $date, $bodyText) {
                    $already = $recipient->isReplied();
                    if (!$already) {
                        $recipient->markAsReplied();
                        if ($campaign) {
                            $campaign->increment('replied_count');
                        }
                    }

                    if ($campaign) {
                        CampaignLog::logEvent(
                            $campaign->id,
                            'replied',
                            $recipient->id,
                            [
                                'email' => $recipient->email,
                                'from' => $from,
                                'subject' => $subject,
                                'message_id' => $messageId,
                            ]
                        );

                        $parsedFrom = $this->parseFromHeader($from);
                        $receivedAt = $this->parseEmailDate($date);

                        $payload = [
                            'campaign_id' => $campaign->id,
                            'recipient_id' => $recipient->id,
                            'message_id' => $messageKey !== '' ? $messageKey : null,
                            'from_email' => $parsedFrom['email'] ?? null,
                            'from_name' => $parsedFrom['name'] ?? null,
                            'subject' => $subject,
                            'body_text' => $bodyText,
                            'received_at' => $receivedAt,
                        ];

                        if ($messageKey !== '') {
                            CampaignReply::updateOrCreate(
                                ['campaign_id' => $campaign->id, 'message_id' => $messageKey],
                                $payload
                            );
                        } else {
                            CampaignReply::create($payload);
                        }
                    }
                });

                if (!$wasReplied && $campaign) {
                    $subscriber = ListSubscriber::query()
                        ->where('list_id', $campaign->list_id)
                        ->where('email', strtolower(trim((string) ($recipient->email ?? ''))))
                        ->first();

                    if ($subscriber) {
                        try {
                            app(AutomationTriggerService::class)->triggerSubscriberEvent('campaign_replied', $subscriber, [
                                'campaign_id' => $campaign->id,
                                'recipient_id' => $recipient->id,
                            ]);
                        } catch (\Throwable $e) {
                            Log::warning('Failed to trigger automation on campaign_replied', [
                                'campaign_id' => $campaign->id,
                                'recipient_id' => $recipient->id,
                                'error' => $e->getMessage(),
                            ]);
                        }
                    }
                }

                if ($campaign) {
                    $this->forwardReply($campaign, $recipient, $raw, $from, $subject);
                }

                $processed++;

                imap_setflag_full($connection, (string) $messageNumber, "\\Seen");
                if ($deleteAfter) {
                    imap_delete($connection, $messageNumber);
                }
            } catch (\Exception $e) {
                Log::error('Error processing reply message: ' . $e->getMessage(), [
                    'message_number' => $messageNumber,
                ]);
            }
        }

        if ($deleteAfter) {
            imap_expunge($connection);
        }

        imap_close($connection);

        return $processed;
    }

    protected function resolveImapSettings(?ReplyServer $replyServer = null): array
    {
        if ($replyServer) {
            return [
                'hostname' => (string) ($replyServer->hostname ?? ''),
                'username' => (string) ($replyServer->username ?? ''),
                'password' => (string) ($replyServer->password ?? ''),
                'mailbox' => (string) ($replyServer->mailbox ?? 'INBOX'),
                'port' => (int) ($replyServer->port ?? 993),
                'encryption' => (string) ($replyServer->encryption ?? 'ssl'),
                'delete_after_processing' => (bool) ($replyServer->delete_after_processing ?? false),
                'max_emails_per_batch' => (int) ($replyServer->max_emails_per_batch ?? 50),
            ];
        }

        return (array) config('mailpurse.reply_tracking.imap', []);
    }

    protected function forwardReply(Campaign $campaign, CampaignRecipient $recipient, string $raw, ?string $from, ?string $subject): void
    {
        $destination = (string) ($campaign->reply_to ?? '');
        if ($destination === '') {
            $destination = (string) ($campaign->from_email ?? config('mail.from.address'));
        }

        if ($destination === '') {
            return;
        }

        $body = $this->extractPlainBody($raw);

        $forwardSubject = 'FWD: ' . (string) ($subject ?? 'Campaign reply');

        try {
            Mail::raw($body, function ($message) use ($destination, $forwardSubject, $from, $campaign, $recipient) {
                $message->to($destination)
                    ->subject($forwardSubject);

                if (is_string($from) && trim($from) !== '') {
                    $message->replyTo($from);
                }

                $message->getHeaders()->addTextHeader('X-MailPurse-Reply-Forwarded', '1');
                $message->getHeaders()->addTextHeader('X-Campaign-ID', (string) $campaign->id);
                $message->getHeaders()->addTextHeader('X-Recipient-UUID', (string) $recipient->uuid);
            });
        } catch (\Throwable $e) {
            Log::warning('Failed to forward reply', [
                'campaign_id' => $campaign->id,
                'recipient_uuid' => $recipient->uuid,
                'error' => $e->getMessage(),
            ]);
        }
    }

    protected function isForwardedMessage(string $raw): bool
    {
        return stripos($raw, 'X-MailPurse-Reply-Forwarded:') !== false;
    }

    protected function extractRecipientUuid(string $raw): ?string
    {
        $candidates = [
            $this->extractHeaderValue($raw, 'To'),
            $this->extractHeaderValue($raw, 'Delivered-To'),
            $this->extractHeaderValue($raw, 'X-Original-To'),
        ];

        foreach ($candidates as $value) {
            if (!is_string($value) || trim($value) === '') {
                continue;
            }

            if (preg_match('/reply\+([0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12})@/i', $value, $m)) {
                return strtolower($m[1]);
            }
        }

        if (preg_match('/\b([0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12})\b/i', $raw, $m)) {
            return strtolower($m[1]);
        }

        return null;
    }

    protected function extractHeaderValue(string $raw, string $headerName): ?string
    {
        $pattern = '/^' . preg_quote($headerName, '/') . ':(.*)$/im';
        if (!preg_match($pattern, $raw, $matches)) {
            return null;
        }

        $value = trim((string) ($matches[1] ?? ''));

        return $value !== '' ? $value : null;
    }

    protected function extractPlainBody(string $raw): string
    {
        $parts = preg_split("/\r\n\r\n/", $raw, 2);
        $body = $parts[1] ?? '';

        $body = trim((string) $body);

        if ($body === '') {
            return '';
        }

        if (stripos($body, '<html') !== false) {
            $body = strip_tags($body);
        }

        return $body;
    }

    protected function normalizeMessageId(?string $messageId): string
    {
        if (!is_string($messageId)) {
            return '';
        }

        $messageId = trim($messageId);
        $messageId = trim($messageId, "<> \t\n\r\0\x0B");

        return $messageId;
    }

    protected function parseFromHeader(?string $from): array
    {
        $from = trim((string) ($from ?? ''));
        if ($from === '') {
            return ['email' => null, 'name' => null];
        }

        if (preg_match('/^(.*)<\s*([^>]+)\s*>$/', $from, $m)) {
            $name = trim((string) ($m[1] ?? ''));
            $email = trim((string) ($m[2] ?? ''));
            $name = trim($name, "\"' \t\n\r\0\x0B");

            return [
                'email' => $email !== '' ? $email : null,
                'name' => $name !== '' ? $name : null,
            ];
        }

        return [
            'email' => $from,
            'name' => null,
        ];
    }

    protected function parseEmailDate(?string $date): ?Carbon
    {
        $date = trim((string) ($date ?? ''));
        if ($date === '') {
            return null;
        }

        try {
            return Carbon::parse($date);
        } catch (\Throwable) {
            return null;
        }
    }
}
