<?php

namespace App\Services\EmailValidation;

use Illuminate\Support\Facades\Http;

class SnapvalidClient
{
    public function verify(string $apiKey, string $email): array
    {
        $email = trim($email);

        try {
            $response = Http::timeout(20)
                ->retry(2, 200)
                ->get('https://app.snapvalid.com/api/v1/verify', [
                    'apikey' => $apiKey,
                    'email' => $email,
                ]);

            if (!$response->successful()) {
                return [
                    'success' => false,
                    'result' => null,
                    'message' => 'Snapvalid request failed.',
                    'flags' => null,
                    'raw' => [
                        'status' => $response->status(),
                        'body' => $response->body(),
                    ],
                ];
            }

            $data = $response->json();
            if (!is_array($data)) {
                return [
                    'success' => false,
                    'result' => null,
                    'message' => 'Snapvalid returned invalid JSON.',
                    'flags' => null,
                    'raw' => ['body' => $response->body()],
                ];
            }

            $success = (bool) ($data['success'] ?? false);
            $result = is_string($data['result'] ?? null) ? $data['result'] : null;
            if (is_string($result)) {
                $result = strtolower(trim($result));
                $result = str_replace('-', '_', $result);
            }

            $flags = array_intersect_key($data, array_flip([
                'accept_all',
                'role',
                'free_email',
                'disposable',
                'spamtrap',
            ]));

            return [
                'success' => $success,
                'result' => $result,
                'message' => is_string($data['message'] ?? null) ? $data['message'] : null,
                'flags' => $flags,
                'raw' => $data,
            ];
        } catch (\Throwable $e) {
            return [
                'success' => false,
                'result' => null,
                'message' => $e->getMessage(),
                'flags' => null,
                'raw' => null,
            ];
        }
    }
}
