<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Support\Facades\Crypt;

class BounceServer extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'customer_id',
        'name',
        'protocol',
        'hostname',
        'port',
        'encryption',
        'username',
        'password',
        'mailbox',
        'active',
        'delete_after_processing',
        'max_emails_per_batch',
        'notes',
    ];

    protected function casts(): array
    {
        return [
            'active' => 'boolean',
            'delete_after_processing' => 'boolean',
            'port' => 'integer',
            'max_emails_per_batch' => 'integer',
        ];
    }

    /**
     * Encrypt password when setting.
     */
    public function setPasswordAttribute($value): void
    {
        $this->attributes['password'] = Crypt::encryptString($value);
    }

    /**
     * Decrypt password when getting.
     */
    public function getPasswordAttribute($value): string
    {
        try {
            return Crypt::decryptString($value);
        } catch (\Exception $e) {
            return $value; // Return as-is if decryption fails
        }
    }

    public function customer(): BelongsTo
    {
        return $this->belongsTo(Customer::class);
    }

    public function bounceLogs(): HasMany
    {
        return $this->hasMany(BounceLog::class);
    }

    public function isActive(): bool
    {
        return $this->active;
    }
}
