<?php

namespace App\Mail;

use App\Models\Campaign;
use App\Models\CampaignRecipient;
use App\Models\SendingDomain;
use App\Services\ReplyTrackingAddressService;
use Illuminate\Bus\Queueable;
use Illuminate\Mail\Mailable;
use Illuminate\Mail\Mailables\Address;
use Illuminate\Mail\Mailables\Content;
use Illuminate\Mail\Mailables\Envelope;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;

class CampaignMailable extends Mailable
{
    use Queueable, SerializesModels;

    /**
     * Create a new message instance.
     */
    public function __construct(
        public Campaign $campaign,
        public CampaignRecipient $recipient
    ) {
        // Set Return-Path header using withSymfonyMessage
        $this->setReturnPathHeader();
        $this->setTrackingHeaders();
    }
    
    /**
     * Set the Return-Path header to bounce server email.
     */
    protected function setReturnPathHeader(): void
    {
        $returnPath = $this->getReturnPathEmail();
        
        if ($returnPath) {
            $this->withSymfonyMessage(function ($message) use ($returnPath) {
                // Use returnPath() method on Symfony Email object
                $message->returnPath($returnPath);
            });
            
            Log::info('Set Return-Path header in campaign email', [
                'campaign_id' => $this->campaign->id,
                'recipient_email' => $this->recipient->email ?? 'unknown',
                'return_path' => $returnPath,
            ]);
        }
    }

    protected function setTrackingHeaders(): void
    {
        $campaignId = $this->campaign->id;
        $listId = $this->campaign->list_id;
        $recipientUuid = $this->recipient->uuid;

        $this->withSymfonyMessage(function ($message) use ($campaignId, $listId, $recipientUuid) {
            $headers = $message->getHeaders();
            $headers->addTextHeader('X-Campaign-ID', (string) $campaignId);
            if (!empty($listId)) {
                $headers->addTextHeader('X-List-ID', (string) $listId);
            }
            $headers->addTextHeader('X-Recipient-UUID', (string) $recipientUuid);
        });

        Log::debug('Added campaign tracking headers to email', [
            'campaign_id' => $campaignId,
            'list_id' => $listId,
            'recipient_id' => $this->recipient->id,
            'recipient_uuid' => $recipientUuid,
        ]);
    }

    /**
     * Get the message envelope.
     */
    public function envelope(): Envelope
    {
        // Get from email, using sending domain if available
        $fromEmail = $this->getFromEmail();
        $fromName = $this->campaign->from_name ?? null;
        
        // Ensure fromEmail is a string (handle case where it might be an array)
        if (is_array($fromEmail)) {
            $fromEmail = $fromEmail['address'] ?? $fromEmail[0] ?? (string) reset($fromEmail);
        }
        $fromEmail = (string) $fromEmail;
        
        // Ensure fromName is a string if provided
        if ($fromName && is_array($fromName)) {
            $fromName = $fromName['name'] ?? $fromName[0] ?? (string) reset($fromName);
        }
        $fromName = $fromName ? (string) $fromName : null;
        
        // Create Address object if name is provided, otherwise use string
        $from = $fromName 
            ? new Address($fromEmail, $fromName)
            : $fromEmail;

        $replyTo = app(ReplyTrackingAddressService::class)
            ->effectiveReplyTo($this->campaign, $this->recipient, $fromEmail);

        // Log bounce server username for each email being sent
        $this->logBounceServerInfo();
        
        return new Envelope(
            subject: $this->personalizeContent($this->campaign->subject),
            from: $from,
            replyTo: $replyTo,
        );
    }
    
    /**
     * Log bounce server information for this email.
     */
    protected function logBounceServerInfo(): void
    {
        try {
            // Load delivery server with bounce server relationship
            $this->campaign->loadMissing('deliveryServer.bounceServer');
            
            $deliveryServer = $this->campaign->deliveryServer;
            $bounceServer = $deliveryServer?->bounceServer;
            
            Log::info('Sending campaign email with bounce server info', [
                'campaign_id' => $this->campaign->id,
                'recipient_email' => $this->recipient->email,
                'recipient_id' => $this->recipient->id,
                'delivery_server_id' => $deliveryServer?->id,
                'delivery_server_name' => $deliveryServer?->name,
                'bounce_server_id' => $bounceServer?->id,
                'bounce_server_username' => $bounceServer?->username,
                'bounce_server_email' => $bounceServer?->username, // Alias for clarity
            ]);
        } catch (\Exception $e) {
            // Log error but don't fail email sending
            Log::warning('Failed to log bounce server info', [
                'campaign_id' => $this->campaign->id,
                'recipient_email' => $this->recipient->email,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Get the from email address, using sending domain if available.
     */
    protected function getFromEmail(): string
    {
        // Priority: Campaign's sending domain > Email list's sending domain > Default
        $sendingDomain = null;
        
        // First, check if campaign has a sending domain
        if ($this->campaign->sendingDomain && $this->campaign->sendingDomain->isVerified()) {
            $sendingDomain = $this->campaign->sendingDomain;
        }
        // Fallback to email list's sending domain
        elseif ($this->campaign->emailList && $this->campaign->emailList->sendingDomain && $this->campaign->emailList->sendingDomain->isVerified()) {
            $sendingDomain = $this->campaign->emailList->sendingDomain;
        }
        
        // Get the base from email, ensuring it's a string
        $baseFromEmail = $this->campaign->from_email 
            ?? $this->campaign->emailList->from_email 
            ?? $this->getDefaultFromEmail();
        
        // Handle case where from_email might be an array
        if (is_array($baseFromEmail)) {
            $baseFromEmail = $baseFromEmail['address'] ?? $baseFromEmail[0] ?? (string) reset($baseFromEmail);
        }
        $baseFromEmail = (string) $baseFromEmail;
        
        // If we have a verified sending domain, use it
        if ($sendingDomain) {
            // Extract the local part (before @) from the from_email
            $parts = explode('@', $baseFromEmail);
            $localPart = $parts[0] ?? 'noreply';
            
            // Use the sending domain
            return $localPart . '@' . $sendingDomain->domain;
        }
        
        // Fallback to campaign's from_email or default
        return $baseFromEmail;
    }
    
    /**
     * Get the default from email from config, ensuring it's a string.
     */
    protected function getDefaultFromEmail(): string
    {
        $mailFrom = config('mail.from');
        
        // Handle different config structures
        if (is_array($mailFrom)) {
            return $mailFrom['address'] ?? $mailFrom[0] ?? 'noreply@example.com';
        }
        
        // If config('mail.from.address') exists, use it
        $mailFromAddress = config('mail.from.address');
        if ($mailFromAddress && !is_array($mailFromAddress)) {
            return (string) $mailFromAddress;
        }
        
        // Final fallback
        return 'noreply@example.com';
    }
    
    /**
     * Get the Return-Path email from bounce server.
     */
    protected function getReturnPathEmail(): ?string
    {
        try {
            // Load delivery server with bounce server relationship
            $this->campaign->loadMissing('deliveryServer.bounceServer');
            
            $deliveryServer = $this->campaign->deliveryServer;
            $bounceServer = $deliveryServer?->bounceServer;
            
            // Return bounce server email if configured
            if ($bounceServer && !empty($bounceServer->username)) {
                return $bounceServer->username;
            }
            
            return null;
        } catch (\Exception $e) {
            Log::warning('Failed to get Return-Path email from bounce server', [
                'campaign_id' => $this->campaign->id,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }

    /**
     * Get the message content definition.
     */
    public function content(): Content
    {
        $htmlContent = $this->prepareHtmlContent();
        $plainTextContent = $this->preparePlainTextContent();

        return new Content(
            html: 'emails.campaign',
            text: 'emails.campaign-text',
            with: [
                'campaign' => $this->campaign,
                'recipient' => $this->recipient,
                'htmlContent' => $htmlContent,
                'plainTextContent' => $plainTextContent,
                'trackOpenUrl' => $this->getTrackOpenUrl(),
                'unsubscribeUrl' => $this->getUnsubscribeUrl(),
            ],
        );
    }

    /**
     * Prepare HTML content with tracking and personalization.
     */
    protected function prepareHtmlContent(): string
    {
        $content = $this->personalizeContent($this->campaign->html_content ?? '');

        // Add tracking pixel if enabled
        if ($this->campaign->track_opens) {
            $trackPixel = '<img src="' . $this->getTrackOpenUrl() . '" width="1" height="1" style="display:none;" alt="" />';
            $content = str_replace('</body>', $trackPixel . '</body>', $content);
            if (strpos($content, '</body>') === false) {
                $content .= $trackPixel;
            }
        }

        // Wrap links with click tracking if enabled
        if ($this->campaign->track_clicks) {
            $content = $this->wrapLinksWithTracking($content);
        }

        // Add unsubscribe link
        $unsubscribeLink = '<p style="font-size: 12px; color: #999; margin-top: 20px;">
            <a href="' . $this->getUnsubscribeUrl() . '" style="color: #999;">Unsubscribe</a>
        </p>';
        $content = str_replace('</body>', $unsubscribeLink . '</body>', $content);
        if (strpos($content, '</body>') === false) {
            $content .= $unsubscribeLink;
        }

        return $content;
    }

    /**
     * Prepare plain text content with personalization.
     */
    protected function preparePlainTextContent(): string
    {
        $content = $this->campaign->plain_text_content ?? strip_tags($this->campaign->html_content ?? '');
        $content = $this->personalizeContent($content);
        
        // Add unsubscribe link to plain text
        $content .= "\n\n---\nUnsubscribe: " . $this->getUnsubscribeUrl();

        return $content;
    }

    /**
     * Personalize content with recipient data.
     */
    protected function personalizeContent(string $content): string
    {
        $replacements = [
            '{first_name}' => $this->recipient->first_name ?? '',
            '{last_name}' => $this->recipient->last_name ?? '',
            '{email}' => $this->recipient->email,
            '{full_name}' => trim(($this->recipient->first_name ?? '') . ' ' . ($this->recipient->last_name ?? '')),
        ];

        return str_replace(array_keys($replacements), array_values($replacements), $content);
    }

    /**
     * Wrap links with click tracking.
     */
    protected function wrapLinksWithTracking(string $content): string
    {
        // Match all <a href="..."> tags
        return preg_replace_callback(
            '/<a\s+([^>]*\s+)?href=["\']([^"\']+)["\']([^>]*)>/i',
            function ($matches) {
                $originalUrl = $matches[2];
                $trackUrl = $this->getTrackClickUrl($originalUrl);
                
                // Preserve existing attributes
                $attributes = $matches[1] . $matches[3];
                
                return '<a ' . trim($attributes) . ' href="' . $trackUrl . '">';
            },
            $content
        );
    }

    protected function getTrackingBaseUrl(): ?string
    {
        $this->campaign->loadMissing('trackingDomain');

        if (!$this->campaign->trackingDomain || !$this->campaign->trackingDomain->isVerified()) {
            return null;
        }

        return 'https://' . $this->campaign->trackingDomain->domain;
    }

    protected function trackingRoute(string $name, array $parameters = []): string
    {
        $baseUrl = $this->getTrackingBaseUrl();
        if (!$baseUrl) {
            return route($name, $parameters);
        }

        $path = route($name, $parameters, false);

        return rtrim($baseUrl, '/') . $path;
    }

    /**
     * Get track open URL.
     */
    protected function getTrackOpenUrl(): string
    {
        // Use legacy tracking route which expects the recipient UUID
        // The v2 route requires hashed campaign/subscriber ids that are not available here
        return $this->trackingRoute('track.open.legacy', ['uuid' => $this->recipient->uuid]);
    }

    /**
     * Get track click URL.
     */
    protected function getTrackClickUrl(string $originalUrl): string
    {
        // Use legacy click tracking route that accepts recipient UUID + encoded URL
        $encodedUrl = rtrim(strtr(base64_encode($originalUrl), '+/', '-_'), '=');
        return $this->trackingRoute('track.click.legacy', [
            'uuid' => $this->recipient->uuid,
            'url' => $encodedUrl,
        ]);
    }

    /**
     * Get unsubscribe URL.
     */
    protected function getUnsubscribeUrl(): string
    {
        return $this->trackingRoute('unsubscribe', ['uuid' => $this->recipient->uuid]);
    }
}

