<?php

namespace App\Jobs;

use App\Models\Campaign;
use App\Models\CampaignRecipient;
use App\Models\SuppressionList;
use App\Jobs\SendCampaignChunkJob;
use App\Services\CampaignService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;

class StartCampaignJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $backoff = 60; // 60 seconds between retries

    /**
     * Create a new job instance.
     */
    public function __construct(
        public Campaign $campaign
    ) {}

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // Refresh campaign to get latest status
        $this->campaign->refresh();

        try {
            app(CampaignService::class)->ensureCanRun($this->campaign);
        } catch (\RuntimeException $e) {
            $this->campaign->update([
                'status' => 'failed',
                'finished_at' => now(),
                'failure_reason' => $e->getMessage(),
            ]);
            return;
        }

        if ($this->campaign->scheduled_at && $this->campaign->scheduled_at->isFuture()) {
            $delaySeconds = max(1, now()->diffInSeconds($this->campaign->scheduled_at, false));
            $this->release($delaySeconds);
            return;
        }

        // Check if campaign can still be started
        if (!in_array($this->campaign->status, ['draft', 'scheduled', 'queued'], true)) {
            Log::warning("Campaign {$this->campaign->id} cannot be started. Current status: {$this->campaign->status}");
            return;
        }

        $claimed = DB::transaction(function () {
            return Campaign::whereKey($this->campaign->id)
                ->whereIn('status', ['draft', 'scheduled', 'queued'])
                ->whereNull('started_at')
                ->update([
                    'status' => 'running',
                    'started_at' => now(),
                ]);
        });

        if ($claimed !== 1) {
            Log::warning("Campaign {$this->campaign->id} start was skipped (already started or status changed).", [
                'status' => $this->campaign->status,
                'started_at' => $this->campaign->started_at,
            ]);
            return;
        }

        $this->campaign->refresh();

        try {
            // Prepare recipients if not already prepared
            $this->prepareRecipients();

            // Get pending recipients count
            $pendingCount = $this->campaign->recipients()
                ->where('status', 'pending')
                ->count();

            if ($pendingCount === 0) {
                Log::warning("Campaign {$this->campaign->id} has no pending recipients");
                $this->campaign->update([
                    'status' => 'completed',
                    'finished_at' => now(),
                ]);
                return;
            }

            // Dispatch chunks (50-100 recipients per chunk)
            $chunkSize = 50;
            $this->campaign->recipients()
                ->where('status', 'pending')
                ->chunk($chunkSize, function ($recipients) {
                    SendCampaignChunkJob::dispatch($this->campaign, $recipients->pluck('id')->toArray())
                        ->onQueue('campaigns');
                });

            Log::info("Campaign {$this->campaign->id} started. Dispatched chunks for {$pendingCount} recipients");

        } catch (\Exception $e) {
            $errorMessage = $e->getMessage();
            Log::error("Failed to start campaign {$this->campaign->id}: " . $errorMessage, [
                'exception' => $e,
                'trace' => $e->getTraceAsString()
            ]);
            
            $this->campaign->update([
                'status' => 'failed',
                'finished_at' => now(),
                'failure_reason' => $errorMessage,
            ]);

            throw $e;
        }
    }

    /**
     * Prepare recipients from email list.
     */
    protected function prepareRecipients(): void
    {
        // Check if recipients already exist
        if ($this->campaign->recipients()->count() > 0) {
            return;
        }

        if (!$this->campaign->emailList) {
            throw new \RuntimeException("Campaign {$this->campaign->id} has no associated email list");
        }

        // Get confirmed subscribers from the email list
        // Exclude bounced, complained, and suppressed subscribers
        $subscribers = $this->campaign->emailList->subscribers()
            ->where('status', 'confirmed')
            ->where('is_bounced', false)
            ->where('is_complained', false)
            ->whereNull('suppressed_at')
            ->get();

        // Also check global suppression list
        $suppressedEmails = SuppressionList::where('customer_id', $this->campaign->customer_id)
            ->pluck('email')
            ->toArray();

        if (!empty($suppressedEmails)) {
            $subscribers = $subscribers->reject(function ($subscriber) use ($suppressedEmails) {
                return in_array($subscriber->email, $suppressedEmails);
            });
        }

        // Log subscriber status breakdown for debugging
        $totalSubscribers = $this->campaign->emailList->subscribers()->count();
        $confirmedCount = $this->campaign->emailList->subscribers()->where('status', 'confirmed')->count();
        $unconfirmedCount = $this->campaign->emailList->subscribers()->where('status', 'unconfirmed')->count();
        $unsubscribedCount = $this->campaign->emailList->subscribers()->where('status', 'unsubscribed')->count();
        
        Log::info("Campaign {$this->campaign->id} subscriber breakdown", [
            'total' => $totalSubscribers,
            'confirmed' => $confirmedCount,
            'unconfirmed' => $unconfirmedCount,
            'unsubscribed' => $unsubscribedCount,
        ]);

        if ($subscribers->isEmpty()) {
            throw new \RuntimeException(
                "Email list {$this->campaign->emailList->id} has no confirmed subscribers. " .
                "Total: {$totalSubscribers}, Confirmed: {$confirmedCount}, Unconfirmed: {$unconfirmedCount}, Unsubscribed: {$unsubscribedCount}. " .
                "Please confirm subscribers before starting the campaign."
            );
        }

        // Create recipient records
        $recipients = [];
        foreach ($subscribers as $subscriber) {
            $recipients[] = [
                'campaign_id' => $this->campaign->id,
                'email' => $subscriber->email,
                'uuid' => (string) Str::uuid(),
                'first_name' => $subscriber->first_name,
                'last_name' => $subscriber->last_name,
                'status' => 'pending',
                'created_at' => now(),
                'updated_at' => now(),
            ];
        }

        // Batch insert for performance
        CampaignRecipient::insert($recipients);

        // Update campaign total recipients count
        $this->campaign->update([
            'total_recipients' => count($recipients),
        ]);

        Log::info("Prepared {$this->campaign->total_recipients} recipients for campaign {$this->campaign->id}");
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        $errorMessage = $exception->getMessage();
        Log::error("StartCampaignJob failed for campaign {$this->campaign->id}: " . $errorMessage, [
            'exception' => $exception,
            'trace' => $exception->getTraceAsString()
        ]);
        
        $this->campaign->update([
            'status' => 'failed',
            'finished_at' => now(),
            'failure_reason' => $errorMessage,
        ]);
    }
}

