<?php

namespace App\Jobs;

use App\Mail\CampaignMailable;
use App\Models\Campaign;
use App\Models\CampaignRecipient;
use App\Models\CampaignLog;
use App\Models\DeliveryServer;
use App\Models\SuppressionList;
use App\Services\AutomationTriggerService;
use App\Services\DeliveryServerService;
use App\Services\CampaignService;
use App\Services\ZeptoMailApiService;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\RateLimiter;
use Swift_TransportException;

class SendCampaignChunkJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $backoff = 30; // 30 seconds between retries
    public int $timeout = 300; // 5 minutes timeout

    /**
     * Create a new job instance.
     */
    public function __construct(
        public Campaign $campaign,
        public array $recipientIds
    ) {}

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        // Refresh campaign to get latest status
        $this->campaign->refresh();

        // Check if campaign is still running
        if (!$this->campaign->isRunning()) {
            Log::info("Campaign {$this->campaign->id} is not running (status: {$this->campaign->status}). Skipping chunk.");
            return;
        }

        try {
            app(CampaignService::class)->ensureCanRun($this->campaign);
        } catch (\RuntimeException $e) {
            $this->campaign->update([
                'status' => 'failed',
                'finished_at' => now(),
                'failure_reason' => $e->getMessage(),
            ]);
            return;
        }

        // Get delivery server for campaign (we'll configure mailer before each send)
        $deliveryServer = null;
        $usingFallback = false;

        $this->campaign->loadMissing('customer');
        $customer = $this->campaign->customer;
        $mustAddDelivery = $customer ? (bool) $customer->groupSetting('servers.permissions.must_add_delivery_server', false) : false;
        $canUseSystem = $customer ? (bool) $customer->groupSetting('servers.permissions.can_use_system_servers', false) : false;
        
        if ($this->campaign->delivery_server_id) {
            $deliveryServer = DeliveryServer::with('bounceServer')->find($this->campaign->delivery_server_id);
            
            if (!$deliveryServer || !$deliveryServer->isActive()) {
                Log::warning('Campaign has invalid or inactive delivery server, falling back to default', [
                    'campaign_id' => $this->campaign->id,
                    'configured_delivery_server_id' => $this->campaign->delivery_server_id,
                    'delivery_server_found' => $deliveryServer !== null,
                    'delivery_server_active' => $deliveryServer?->isActive(),
                ]);
                $usingFallback = true;
            }
        } else {
            Log::warning('Campaign has no delivery server configured, using fallback default', [
                'campaign_id' => $this->campaign->id,
                'campaign_name' => $this->campaign->name,
                'message' => 'Campaign delivery_server_id is null. Please select a delivery server in campaign settings.',
            ]);
            $usingFallback = true;
        }
        
        // If not set or invalid, get default active delivery server
        if (!$deliveryServer || !$deliveryServer->isActive()) {
            if ($mustAddDelivery) {
                $this->campaign->update([
                    'status' => 'failed',
                    'finished_at' => now(),
                    'failure_reason' => 'You must select a delivery server before running a campaign.',
                ]);
                return;
            }

            $deliveryServer = DeliveryServer::with('bounceServer')
                ->where('status', 'active')
                ->where('use_for', true)
                ->where(function ($q) use ($customer, $canUseSystem) {
                    if ($customer) {
                        $q->where('customer_id', $customer->id);
                        $q->orWhereNull('customer_id');
                    } else {
                        $q->whereNull('customer_id');
                    }
                })
                ->orderBy('id')
                ->first();

            if (!$deliveryServer) {
                $this->campaign->update([
                    'status' => 'failed',
                    'finished_at' => now(),
                    'failure_reason' => 'No available delivery server for this campaign.',
                ]);
                return;
            }
            
            if ($usingFallback && $deliveryServer) {
                Log::info('Using fallback delivery server for campaign', [
                    'campaign_id' => $this->campaign->id,
                    'fallback_delivery_server_id' => $deliveryServer->id,
                    'fallback_delivery_server_name' => $deliveryServer->name,
                    'configured_delivery_server_id' => $this->campaign->delivery_server_id,
                ]);
            }
        }

        // Get recipients
        $recipients = CampaignRecipient::whereIn('id', $this->recipientIds)
            ->where('campaign_id', $this->campaign->id)
            ->where('status', 'pending')
            ->get();

        // Filter out suppressed recipients
        $suppressedEmails = SuppressionList::where('customer_id', $this->campaign->customer_id)
            ->pluck('email')
            ->toArray();

        $recipients = $recipients->reject(function ($recipient) use ($suppressedEmails) {
            return in_array($recipient->email, $suppressedEmails);
        });

        if ($recipients->isEmpty()) {
            Log::info("No pending recipients found for chunk in campaign {$this->campaign->id}");
            return;
        }

        $sentCount = 0;
        $failedCount = 0;

        if ($deliveryServer && $deliveryServer->type === 'zeptomail-api') {
            $this->sendChunkViaZeptoBatch($deliveryServer, $recipients, $sentCount, $failedCount);

            Log::info("Campaign chunk processed for campaign {$this->campaign->id}: {$sentCount} sent, {$failedCount} failed");

            $this->campaign->refresh();
            $this->campaign->checkCompletion();
            return;
        }

        foreach ($recipients as $recipient) {
            // Double-check campaign status before each send
            $this->campaign->refresh();
            if (!$this->campaign->isRunning()) {
                Log::info("Campaign {$this->campaign->id} was paused/failed. Stopping chunk processing.");
                break;
            }

            try {
                // Configure mailer before each send to ensure fresh configuration
                $mailerName = $this->configureMailer($deliveryServer);
                
                // Rate limiting: 1-2 emails per second
                RateLimiter::attempt(
                    'campaign-send:' . $this->campaign->id,
                    $perMinute = 60, // 60 per minute = 1 per second
                    function () use ($recipient, &$sentCount, $mailerName, $deliveryServer) {
                        // Send email - this should be queued if CampaignMailable implements ShouldQueue
                        // But we're already in a queued job, so we send directly
                        try {
                            // Log bounce server info before sending
                            $bounceServer = $deliveryServer?->bounceServer;
                            Log::info('Sending campaign email', [
                                'campaign_id' => $this->campaign->id,
                                'recipient_email' => $recipient->email,
                                'recipient_id' => $recipient->id,
                                'delivery_server_id' => $deliveryServer?->id,
                                'delivery_server_name' => $deliveryServer?->name,
                                'bounce_server_id' => $bounceServer?->id,
                                'bounce_server_username' => $bounceServer?->username,
                            ]);
                            
                            // Explicitly use the configured mailer to ensure correct delivery server is used
                            $mailer = $mailerName ? Mail::mailer($mailerName) : Mail::mailer();
                            $mailer->to($recipient->email)
                                ->send(new CampaignMailable($this->campaign, $recipient));

                            // Mark as accepted (email was accepted by mail server)
                            // Actual delivery status will be updated via webhooks or later checks
                            DB::transaction(function () use ($recipient) {
                                $recipient->update([
                                    'status' => 'sent', // Keep as 'sent' for now, but log as 'accepted'
                                    'sent_at' => now(),
                                ]);
                                $this->campaign->incrementSentCount();
                                
                                CampaignLog::logEvent(
                                    $this->campaign->id,
                                    'accepted', // Log as 'accepted' to distinguish from actual delivery
                                    $recipient->id,
                                    ['email' => $recipient->email, 'message' => 'Email accepted by mail server']
                                );
                            });

                            try {
                                app(AutomationTriggerService::class)->scheduleNegativeCampaignTriggersForRecipient($this->campaign, $recipient);
                            } catch (\Throwable $e) {
                                Log::warning('Failed scheduling negative campaign automation triggers', [
                                    'campaign_id' => $this->campaign->id,
                                    'recipient_id' => $recipient->id,
                                    'error' => $e->getMessage(),
                                ]);
                            }

                            $sentCount++;
                        } catch (\Swift_TransportException $e) {
                            // Connection/transport errors - immediate failure
                            throw $e;
                        } catch (\Exception $e) {
                            // Other errors - log and mark as failed
                            Log::error("Failed to send campaign email to {$recipient->email}: " . $e->getMessage());
                            
                            DB::transaction(function () use ($recipient, $e) {
                                $recipient->markAsFailed($e->getMessage());
                                $this->campaign->incrementFailedCount();
                                
                                CampaignLog::logEvent(
                                    $this->campaign->id,
                                    'failed',
                                    $recipient->id,
                                    ['email' => $recipient->email],
                                    null,
                                    null,
                                    null,
                                    $e->getMessage()
                                );
                            });
                            
                            throw $e; // Re-throw to be caught by outer catch
                        }
                    }
                );

            } catch (\Exception $e) {
                // Only catch if not already handled above
                if ($recipient->status !== 'failed') {
                    $failedCount++;
                    
                    Log::error("Failed to send campaign email to {$recipient->email}: " . $e->getMessage());

                    DB::transaction(function () use ($recipient, $e) {
                        $recipient->markAsFailed($e->getMessage());
                        $this->campaign->incrementFailedCount();
                        
                        CampaignLog::logEvent(
                            $this->campaign->id,
                            'failed',
                            $recipient->id,
                            ['email' => $recipient->email],
                            null,
                            null,
                            null,
                            $e->getMessage()
                        );
                    });
                }
            }

            // Small delay to respect rate limits
            usleep(500000); // 0.5 seconds between emails
        }

        Log::info("Campaign chunk processed for campaign {$this->campaign->id}: {$sentCount} sent, {$failedCount} failed");

        // Check for bounces after chunk completion if bounce server is configured
        if ($deliveryServer && $deliveryServer->bounceServer) {
            $this->checkBounces($deliveryServer->bounceServer);
        }

        // Check if campaign is complete
        $this->campaign->refresh();
        $this->campaign->checkCompletion();
    }

    /**
     * Check for bounced emails from the bounce server.
     */
    protected function checkBounces($bounceServer): void
    {
        try {
            // Check if IMAP extension is available
            if (!function_exists('imap_open')) {
                Log::warning("IMAP extension not available. Bounce checking skipped for campaign {$this->campaign->id}. Please install php-imap extension.", [
                    'campaign_id' => $this->campaign->id,
                    'bounce_server_id' => $bounceServer->id,
                    'message' => 'Install php-imap extension to enable automatic bounce processing. You can manually process bounces using: php artisan bounces:process',
                ]);
                return;
            }
            
            $bounceProcessor = app(\App\Services\BounceProcessorService::class);
            $processedCount = $bounceProcessor->processBounces($bounceServer);
            
            if ($processedCount > 0) {
                Log::info("Processed {$processedCount} bounce(s) for campaign {$this->campaign->id}", [
                    'campaign_id' => $this->campaign->id,
                    'bounce_server_id' => $bounceServer->id,
                    'processed_count' => $processedCount,
                ]);
                
                // Refresh campaign to get updated bounce count
                $this->campaign->refresh();
                
                // Sync stats to ensure accuracy
                $this->campaign->syncStats();
            }
        } catch (\Exception $e) {
            // Log error but don't fail the chunk job
            Log::error("Failed to check bounces for campaign {$this->campaign->id}: " . $e->getMessage(), [
                'campaign_id' => $this->campaign->id,
                'bounce_server_id' => $bounceServer->id,
                'error' => $e->getMessage(),
            ]);
        }
    }

    /**
     * Configure mailer based on delivery server.
     * Returns the mailer name to use, or null if using default.
     */
    protected function configureMailer(?DeliveryServer $deliveryServer): ?string
    {
        try {
            if (!$deliveryServer || !$deliveryServer->isActive()) {
                Log::warning("No active delivery server found for campaign {$this->campaign->id}. Using default mailer.");
                return null;
            }

            // Ensure bounce server relation is loaded so we can log its email
            $deliveryServer->loadMissing('bounceServer');

            // Determine the effective sending domain for this campaign (if verified)
            $this->campaign->loadMissing('sendingDomain', 'emailList.sendingDomain');
            $sendingDomainOverride = null;
            if ($this->campaign->sendingDomain && $this->campaign->sendingDomain->isVerified()) {
                $sendingDomainOverride = $this->campaign->sendingDomain->domain;
            } elseif ($this->campaign->emailList && $this->campaign->emailList->sendingDomain && $this->campaign->emailList->sendingDomain->isVerified()) {
                $sendingDomainOverride = $this->campaign->emailList->sendingDomain->domain;
            }

            $bounceServer = $deliveryServer->bounceServer;
            $bounceServerId = $bounceServer?->id;
            $bounceEmail = $bounceServer?->username;

            // Log which bounce email is configured for this delivery server
            Log::info('Configuring mailer with delivery/bounce server for campaign', [
                'campaign_id' => $this->campaign->id,
                'delivery_server_id' => $deliveryServer->id,
                'delivery_server_name' => $deliveryServer->name,
                'delivery_server_type' => $deliveryServer->type,
                'delivery_server_from_email' => $this->campaign->from_email,
                'bounce_server_id' => $bounceServerId,
                'bounce_email' => $bounceEmail,
                'has_bounce_server' => $bounceServer !== null,
            ]);

            // Warn if no bounce server is configured
            if (!$bounceServer) {
                Log::warning('Delivery server has no bounce server configured', [
                    'campaign_id' => $this->campaign->id,
                    'delivery_server_id' => $deliveryServer->id,
                    'delivery_server_name' => $deliveryServer->name,
                    'message' => 'Bounced emails will not be properly tracked. Consider configuring a bounce server for this delivery server.',
                ]);
            }

            // Configure mailer using DeliveryServerService
            $deliveryServerService = app(DeliveryServerService::class);
            $deliveryServerService->configureMailFromServer($deliveryServer, $sendingDomainOverride);
            
            // Get the configured mailer name
            $mailerName = config('mail.default', 'smtp');
            
            Log::debug("Configured mailer for campaign {$this->campaign->id} using delivery server: {$deliveryServer->name} ({$deliveryServer->type}), mailer: {$mailerName}");
            
            return $mailerName;
        } catch (\Exception $e) {
            Log::error("Failed to configure mailer for campaign {$this->campaign->id}: " . $e->getMessage());
            // Continue anyway - might use default mailer
            return null;
        }
    }

    protected function sendChunkViaZeptoBatch(DeliveryServer $deliveryServer, $recipients, int &$sentCount, int &$failedCount): void
    {
        $this->campaign->loadMissing('trackingDomain');

        $trackingBase = null;
        if ($this->campaign->trackingDomain && $this->campaign->trackingDomain->isVerified()) {
            $trackingBase = 'https://' . $this->campaign->trackingDomain->domain;
        }
        $trackingBase = $trackingBase ?: rtrim((string) config('app.url'), '/');

        $subject = $this->convertPlaceholdersToZeptoMergeTags((string) ($this->campaign->subject ?? ''));

        $htmlBody = $this->campaign->html_content ?? null;
        $textBody = $this->campaign->plain_text_content ?? null;

        $htmlBody = is_string($htmlBody) ? $this->convertPlaceholdersToZeptoMergeTags($htmlBody) : null;
        $textBody = is_string($textBody) ? $this->convertPlaceholdersToZeptoMergeTags($textBody) : null;

        $unsubscribeUrl = $trackingBase . '/unsubscribe/{{recipient_uuid}}';
        $trackOpenUrl = $trackingBase . '/track/open/{{recipient_uuid}}';

        if (is_string($htmlBody) && $htmlBody !== '') {
            if ($this->campaign->track_opens) {
                $trackPixel = '<img src="' . $trackOpenUrl . '" width="1" height="1" style="display:none;" alt="" />';
                if (stripos($htmlBody, '</body>') !== false) {
                    $htmlBody = str_ireplace('</body>', $trackPixel . '</body>', $htmlBody);
                } else {
                    $htmlBody .= $trackPixel;
                }
            }

            if ($this->campaign->track_clicks) {
                $htmlBody = $this->wrapLinksWithZeptoTracking($htmlBody, $trackingBase);
            }

            $unsubscribeLink = '<p style="font-size: 12px; color: #999; margin-top: 20px; text-align:center;"><a href="' . $unsubscribeUrl . '" style="color: #999;">Unsubscribe</a></p>';
            if (stripos($htmlBody, '{unsubscribe_url}') !== false) {
                $htmlBody = str_ireplace('{unsubscribe_url}', $unsubscribeUrl, $htmlBody);
            } elseif (stripos($htmlBody, '</body>') !== false) {
                $htmlBody = str_ireplace('</body>', $unsubscribeLink . '</body>', $htmlBody);
            } else {
                $htmlBody .= $unsubscribeLink;
            }
        }

        if (!is_string($textBody) || trim($textBody) === '') {
            $textBody = is_string($htmlBody) ? strip_tags($htmlBody) : '';
        }
        $textBody = rtrim((string) $textBody) . "\n\n---\nUnsubscribe: " . $unsubscribeUrl;

        $recipientByEmail = method_exists($recipients, 'keyBy') ? $recipients->keyBy('email') : null;

        $to = [];
        foreach ($recipients as $recipient) {
            $toName = trim((string) (($recipient->first_name ?? '') . ' ' . ($recipient->last_name ?? '')));
            $to[] = [
                'email_address' => [
                    'address' => (string) $recipient->email,
                    'name' => $toName,
                ],
                'merge_info' => [
                    'first_name' => (string) ($recipient->first_name ?? ''),
                    'last_name' => (string) ($recipient->last_name ?? ''),
                    'email' => (string) $recipient->email,
                    'full_name' => $toName,
                    'recipient_uuid' => (string) ($recipient->uuid ?? ''),
                ],
            ];
        }

        $message = [
            'from_email' => (string) ($this->campaign->from_email ?? ''),
            'from_name' => (string) ($this->campaign->from_name ?? ''),
            'subject' => $subject,
            'htmlbody' => $htmlBody,
            'textbody' => $textBody,
            'track_clicks' => (bool) ($this->campaign->track_clicks ?? false),
            'track_opens' => (bool) ($this->campaign->track_opens ?? false),
            'client_reference' => 'campaign-' . $this->campaign->id . '-{{recipient_uuid}}',
            'headers' => [
                'X-Campaign-ID' => (string) $this->campaign->id,
                'X-List-ID' => (string) ($this->campaign->list_id ?? ''),
                'X-Delivery-Server-ID' => (string) ($deliveryServer->id ?? ''),
            ],
        ];

        $replyTrackingEnabled = (bool) config('mailpurse.reply_tracking.enabled', false);

        $this->campaign->loadMissing('replyServer');
        $replyDomain = trim((string) ($this->campaign->replyServer?->reply_domain ?? config('mailpurse.reply_tracking.reply_domain', '')));
        if ($replyTrackingEnabled && $replyDomain !== '') {
            $message['headers']['Reply-To'] = 'reply+{{recipient_uuid}}@' . $replyDomain;
        }

        $service = app(ZeptoMailApiService::class);
        $chunks = array_chunk($to, 500);

        foreach ($chunks as $chunk) {
            try {
                $service->sendBatch($deliveryServer, $message, $chunk);

                DB::transaction(function () use ($chunk, $recipientByEmail, &$sentCount) {
                    foreach ($chunk as $toItem) {
                        $email = $toItem['email_address']['address'] ?? null;
                        if (!is_string($email) || $email === '') {
                            continue;
                        }

                        $recipient = null;
                        if ($recipientByEmail) {
                            $recipient = $recipientByEmail->get($email);
                        }

                        if (!$recipient) {
                            $recipient = CampaignRecipient::where('campaign_id', $this->campaign->id)
                                ->where('email', $email)
                                ->where('status', 'pending')
                                ->first();
                        }

                        if (!$recipient || $recipient->status !== 'pending') {
                            continue;
                        }

                        $recipient->update([
                            'status' => 'sent',
                            'sent_at' => now(),
                        ]);
                        $this->campaign->incrementSentCount();

                        CampaignLog::logEvent(
                            $this->campaign->id,
                            'accepted',
                            $recipient->id,
                            ['email' => $recipient->email, 'message' => 'Email accepted by ZeptoMail API']
                        );

                        $sentCount++;

                        try {
                            app(AutomationTriggerService::class)->scheduleNegativeCampaignTriggersForRecipient($this->campaign, $recipient);
                        } catch (\Throwable $e) {
                            Log::warning('Failed scheduling negative campaign automation triggers', [
                                'campaign_id' => $this->campaign->id,
                                'recipient_id' => $recipient->id,
                                'error' => $e->getMessage(),
                            ]);
                        }
                    }
                });
            } catch (\Exception $e) {
                $failedCount += count($chunk);
                Log::error('ZeptoMail batch send failed: ' . $e->getMessage(), [
                    'campaign_id' => $this->campaign->id,
                    'delivery_server_id' => $deliveryServer->id,
                ]);

                DB::transaction(function () use ($chunk, $e) {
                    foreach ($chunk as $toItem) {
                        $email = $toItem['email_address']['address'] ?? null;
                        if (!is_string($email) || $email === '') {
                            continue;
                        }

                        $recipient = CampaignRecipient::where('campaign_id', $this->campaign->id)
                            ->where('email', $email)
                            ->where('status', 'pending')
                            ->first();

                        if (!$recipient) {
                            continue;
                        }

                        $recipient->markAsFailed($e->getMessage());
                        $this->campaign->incrementFailedCount();

                        CampaignLog::logEvent(
                            $this->campaign->id,
                            'failed',
                            $recipient->id,
                            ['email' => $recipient->email],
                            null,
                            null,
                            null,
                            $e->getMessage()
                        );
                    }
                });
            }
        }
    }

    protected function convertPlaceholdersToZeptoMergeTags(string $content): string
    {
        return str_replace(
            ['{first_name}', '{last_name}', '{email}', '{full_name}'],
            ['{{first_name}}', '{{last_name}}', '{{email}}', '{{full_name}}'],
            $content
        );
    }

    protected function wrapLinksWithZeptoTracking(string $content, string $trackingBase): string
    {
        return preg_replace_callback(
            '/<a\s+([^>]*\s+)?href=["\']([^"\']+)["\']([^>]*)>/i',
            function ($matches) use ($trackingBase) {
                $originalUrl = $matches[2];
                $encodedUrl = rtrim(strtr(base64_encode($originalUrl), '+/', '-_'), '=');
                $trackUrl = rtrim($trackingBase, '/') . '/track/click/{{recipient_uuid}}/' . $encodedUrl;
                $attributes = $matches[1] . $matches[3];
                return '<a ' . trim($attributes) . ' href="' . $trackUrl . '">';
            },
            $content
        );
    }

    /**
     * Handle a job failure.
     */
    public function failed(\Throwable $exception): void
    {
        Log::error("SendCampaignChunkJob failed for campaign {$this->campaign->id}: " . $exception->getMessage());
        
        // Mark recipients as failed
        CampaignRecipient::whereIn('id', $this->recipientIds)
            ->where('status', 'pending')
            ->update([
                'status' => 'failed',
                'failed_at' => now(),
                'failure_reason' => $exception->getMessage(),
            ]);
    }
}

