<?php

namespace App\Jobs;

use App\Models\EmailValidationRun;
use App\Models\EmailValidationRunItem;
use App\Models\ListSubscriber;
use App\Models\SuppressionList;
use App\Models\UsageLog;
use App\Services\EmailValidation\SnapvalidClient;
use App\Services\ListSubscriberService;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;

class ProcessEmailValidationChunkJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public int $tries = 3;
    public int $backoff = 30;
    public int $timeout = 300;

    /**
     * @param  array<int>  $subscriberIds
     */
    public function __construct(
        public EmailValidationRun $run,
        public array $subscriberIds
    ) {
    }

    public function handle(
        SnapvalidClient $snapvalidClient,
        ListSubscriberService $subscriberService
    ): void {
        $this->run->refresh();

        if ($this->run->status !== 'running') {
            return;
        }

        $this->run->loadMissing(['customer', 'tool']);
        $customer = $this->run->customer;
        $tool = $this->run->tool;

        if (!$customer || !$tool || !$tool->active) {
            $this->failRun('Email validation tool is missing or inactive.');
            return;
        }

        $monthlyLimit = (int) $customer->groupSetting('email_validation.monthly_limit', 0);

        $now = Carbon::now();
        $periodStart = $now->copy()->startOfMonth()->toDateString();
        $periodEnd = $now->copy()->endOfMonth()->toDateString();

        $subscribers = ListSubscriber::query()
            ->where('list_id', $this->run->list_id)
            ->whereIn('id', $this->subscriberIds)
            ->get(['id', 'email']);

        if ($subscribers->isEmpty()) {
            return;
        }

        $emails = $subscribers->pluck('email')
            ->filter()
            ->map(fn ($e) => strtolower(trim((string) $e)))
            ->values()
            ->all();

        $existingItems = EmailValidationRunItem::query()
            ->where('run_id', $this->run->id)
            ->whereIn('email', $emails)
            ->get()
            ->keyBy(fn ($item) => strtolower(trim((string) $item->email)));

        foreach ($subscribers as $subscriber) {
            $email = strtolower(trim((string) $subscriber->email));
            if ($email === '') {
                continue;
            }

            $existingItem = $existingItems->get($email);
            $success = null;
            $normalizedResult = null;
            $message = null;
            $flags = null;
            $raw = null;

            if ($existingItem) {
                $success = (bool) $existingItem->success;
                $normalizedResult = is_string($existingItem->result) ? $existingItem->result : null;
                $message = $existingItem->message;
                $flags = $existingItem->flags;
                $raw = $existingItem->raw;
            } else {
                if ($tool->provider === 'snapvalid') {
                    $verification = $snapvalidClient->verify((string) $tool->api_key, $email);
                } else {
                    $verification = [
                        'success' => false,
                        'result' => null,
                        'message' => 'Unsupported provider.',
                        'flags' => null,
                        'raw' => null,
                    ];
                }

                $success = (bool) ($verification['success'] ?? false);
                $result = is_string($verification['result'] ?? null) ? strtolower(trim($verification['result'])) : null;

                $normalizedResult = match ($result) {
                    'deliverable' => 'deliverable',
                    'undeliverable' => 'undeliverable',
                    'accept_all', 'accept-all' => 'accept_all',
                    'unknown' => 'unknown',
                    default => ($success ? 'unknown' : null),
                };

                $message = is_string($verification['message'] ?? null) ? $verification['message'] : null;
                $flags = is_array($verification['flags'] ?? null) ? $verification['flags'] : null;
                $raw = is_array($verification['raw'] ?? null) ? $verification['raw'] : null;

                $actionTaken = 'none';
                $deliverableDelta = $success && $normalizedResult === 'deliverable' ? 1 : 0;
                $undeliverableDelta = $success && $normalizedResult === 'undeliverable' ? 1 : 0;
                $acceptAllDelta = $success && $normalizedResult === 'accept_all' ? 1 : 0;
                $unknownDelta = $success && in_array($normalizedResult, ['unknown', null], true) ? 1 : 0;
                $errorDelta = $success ? 0 : 1;

                $createdItem = null;
                try {
                    DB::transaction(function () use (
                        $subscriber,
                        $email,
                        $success,
                        $normalizedResult,
                        $message,
                        $flags,
                        $raw,
                        $actionTaken,
                        $customer,
                        $monthlyLimit,
                        $periodStart,
                        $periodEnd,
                        $deliverableDelta,
                        $undeliverableDelta,
                        $acceptAllDelta,
                        $unknownDelta,
                        $errorDelta,
                        &$createdItem
                    ) {
                        $log = UsageLog::where('customer_id', $customer->id)
                            ->where('metric', 'email_validation_emails_this_month')
                            ->where('period_start', $periodStart)
                            ->where('period_end', $periodEnd)
                            ->lockForUpdate()
                            ->first();

                        if (!$log) {
                            UsageLog::create([
                                'customer_id' => $customer->id,
                                'metric' => 'email_validation_emails_this_month',
                                'amount' => 0,
                                'period_start' => $periodStart,
                                'period_end' => $periodEnd,
                                'context' => [],
                            ]);

                            $log = UsageLog::where('customer_id', $customer->id)
                                ->where('metric', 'email_validation_emails_this_month')
                                ->where('period_start', $periodStart)
                                ->where('period_end', $periodEnd)
                                ->lockForUpdate()
                                ->first();
                        }

                        if ($monthlyLimit > 0 && ((int) $log->amount + 1) > $monthlyLimit) {
                            throw new \RuntimeException('Monthly validation limit exceeded.');
                        }

                        $log->increment('amount', 1);
                        $log->update([
                            'context' => array_merge($log->context ?? [], ['run_id' => $this->run->id]),
                        ]);

                        $createdItem = EmailValidationRunItem::create([
                            'run_id' => $this->run->id,
                            'subscriber_id' => $subscriber->id,
                            'email' => $email,
                            'success' => (bool) $success,
                            'result' => $normalizedResult,
                            'message' => $message,
                            'action_taken' => $actionTaken,
                            'flags' => $flags,
                            'raw' => $raw,
                            'validated_at' => now(),
                        ]);

                        EmailValidationRun::whereKey($this->run->id)->update([
                            'processed_count' => DB::raw('processed_count + 1'),
                            'deliverable_count' => DB::raw('deliverable_count + ' . (int) $deliverableDelta),
                            'undeliverable_count' => DB::raw('undeliverable_count + ' . (int) $undeliverableDelta),
                            'accept_all_count' => DB::raw('accept_all_count + ' . (int) $acceptAllDelta),
                            'unknown_count' => DB::raw('unknown_count + ' . (int) $unknownDelta),
                            'error_count' => DB::raw('error_count + ' . (int) $errorDelta),
                        ]);
                    });
                } catch (\RuntimeException $e) {
                    $this->failRun($e->getMessage());
                    break;
                }

                $existingItem = $createdItem;
                $existingItems->put($email, $createdItem);
            }

            if (
                $existingItem
                && $existingItem->action_taken === 'none'
                && $existingItem->success
                && $existingItem->result === 'undeliverable'
            ) {
                $action = (string) ($this->run->invalid_action ?? 'none');
                $actionTaken = 'none';

                if ($action === 'unsubscribe') {
                    $subscriberModel = ListSubscriber::find($subscriber->id);
                    if ($subscriberModel) {
                        $subscriberService->unsubscribe($subscriberModel);
                        $actionTaken = 'unsubscribe';
                    }
                } elseif ($action === 'mark_spam') {
                    $subscriberModel = ListSubscriber::find($subscriber->id);
                    if ($subscriberModel) {
                        $subscriberService->unsubscribe($subscriberModel);
                        $subscriberModel->update([
                            'is_complained' => true,
                            'suppressed_at' => now(),
                        ]);

                        SuppressionList::firstOrCreate(
                            [
                                'customer_id' => $customer->id,
                                'email' => $email,
                            ],
                            [
                                'reason' => 'complaint',
                                'reason_description' => 'Marked as spam by email validation.',
                                'subscriber_id' => $subscriberModel->id,
                                'campaign_id' => null,
                                'suppressed_at' => now(),
                            ]
                        );

                        $actionTaken = 'mark_spam';
                    }
                } elseif ($action === 'delete') {
                    $subscriberModel = ListSubscriber::find($subscriber->id);
                    if ($subscriberModel) {
                        $subscriberService->delete($subscriberModel);
                        $actionTaken = 'delete';
                    }
                }

                if ($actionTaken !== 'none') {
                    $existingItem->update([
                        'action_taken' => $actionTaken,
                    ]);
                }
            }
        }

        $this->run->refresh();

        if ($this->run->status === 'running' && $this->run->processed_count >= $this->run->total_emails) {
            $this->run->update([
                'status' => 'completed',
                'finished_at' => now(),
            ]);
        }
    }

    private function failRun(string $reason): void
    {
        $this->run->refresh();

        if (in_array($this->run->status, ['completed', 'failed'], true)) {
            return;
        }

        $this->run->update([
            'status' => 'failed',
            'finished_at' => now(),
            'failure_reason' => $reason,
        ]);
    }

    public function failed(\Throwable $exception): void
    {
        Log::error('Email validation chunk job failed', [
            'run_id' => $this->run->id,
            'message' => $exception->getMessage(),
        ]);

        $this->failRun($exception->getMessage());
    }
}
