<?php

namespace App\Http\Controllers\Webhook;

use App\Http\Controllers\Controller;
use App\Models\CampaignLog;
use App\Models\CampaignRecipient;
use App\Models\ListSubscriber;
use App\Models\SuppressionList;
use App\Services\ComplaintService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class SesWebhookController extends Controller
{
    public function __construct(
        protected ComplaintService $complaintService
    ) {}

    /**
     * Handle Amazon SES webhook (SNS notification).
     */
    public function handle(Request $request)
    {
        // SES sends notifications via SNS
        $messageType = $request->header('x-amz-sns-message-type');

        if ($messageType === 'SubscriptionConfirmation') {
            // Confirm subscription
            return $this->confirmSubscription($request);
        }

        if ($messageType === 'Notification') {
            return $this->handleNotification($request);
        }

        return response()->json(['status' => 'ignored'], 200);
    }

    /**
     * Confirm SNS subscription.
     */
    protected function confirmSubscription(Request $request): \Illuminate\Http\JsonResponse
    {
        $subscribeUrl = $request->input('SubscribeURL');
        
        if ($subscribeUrl) {
            // In production, you should verify the signature first
            // For now, just log it
            Log::info('SES subscription confirmation received', [
                'url' => $subscribeUrl,
            ]);
        }

        return response()->json(['status' => 'confirmed'], 200);
    }

    /**
     * Handle SNS notification.
     */
    protected function handleNotification(Request $request): \Illuminate\Http\JsonResponse
    {
        // Validate SNS signature (simplified - should verify in production)
        $message = json_decode($request->input('Message'), true);

        if (!$message) {
            return response()->json(['error' => 'Invalid message'], 400);
        }

        // Support SES Event Publishing style payloads (eventType: Open/Click/etc.)
        $eventType = $message['eventType'] ?? null;
        if (is_string($eventType) && $eventType !== '') {
            if (strtolower($eventType) === 'open') {
                $this->handleOpen($request, $message);
                return response()->json(['status' => 'processed'], 200);
            }

            if (strtolower($eventType) === 'click') {
                $this->handleClick($request, $message);
                return response()->json(['status' => 'processed'], 200);
            }
        }

        $notificationType = $message['notificationType'] ?? null;

        // Handle complaint
        if ($notificationType === 'Complaint') {
            return $this->handleComplaint($message);
        }

        // Handle bounce (optional)
        if ($notificationType === 'Bounce') {
            return $this->handleBounce($message);
        }

        return response()->json(['status' => 'ignored'], 200);
    }

    /**
     * Handle complaint notification.
     */
    protected function handleComplaint(array $message): \Illuminate\Http\JsonResponse
    {
        $mail = $message['mail'] ?? [];
        $complaint = $message['complaint'] ?? [];

        $recipients = $complaint['complainedRecipients'] ?? [];
        
        if (empty($recipients)) {
            return response()->json(['error' => 'No recipients'], 400);
        }

        try {
            foreach ($recipients as $recipient) {
                $email = $recipient['emailAddress'] ?? null;
                
                if (!$email) {
                    continue;
                }

                $this->complaintService->processComplaint(
                    email: $email,
                    provider: 'ses',
                    providerMessageId: $mail['messageId'] ?? null,
                    feedbackId: $complaint['feedbackId'] ?? null,
                    rawData: json_encode($message),
                    meta: $message
                );
            }

            return response()->json(['status' => 'processed'], 200);
        } catch (\Exception $e) {
            Log::error('Error processing SES complaint: ' . $e->getMessage(), [
                'message' => $message,
            ]);

            return response()->json(['error' => 'Processing failed'], 500);
        }
    }

    /**
     * Handle bounce notification (optional).
     */
    protected function handleBounce(array $message): \Illuminate\Http\JsonResponse
    {
        $recipient = $this->resolveRecipientFromSnsBouncePayload($message);
        if (!$recipient) {
            Log::info('SES bounce event received but no matching recipient found');
            return response()->json(['status' => 'ignored'], 200);
        }

        $campaign = $recipient->campaign;
        $bounce = $message['bounce'] ?? [];

        $bounceType = strtolower((string) ($bounce['bounceType'] ?? ''));
        $isHardBounce = $bounceType === 'permanent' || $bounceType === 'undetermined';

        $bouncedRecipients = $bounce['bouncedRecipients'] ?? [];
        $reason = null;
        if (is_array($bouncedRecipients) && isset($bouncedRecipients[0]) && is_array($bouncedRecipients[0])) {
            $reason = $bouncedRecipients[0]['diagnosticCode'] ?? $bouncedRecipients[0]['status'] ?? null;
        }
        $reason = (string) ($reason ?? 'SES reported bounce');

        $alreadyBounced = $recipient->status === 'bounced';
        if (!$alreadyBounced) {
            $recipient->markAsBounced();
            if ($campaign) {
                $campaign->incrementBouncedCount();
            }

            CampaignLog::logEvent(
                $campaign?->id ?? 0,
                'bounced',
                $recipient->id,
                [
                    'email' => $recipient->email,
                    'provider' => 'ses',
                    'bounce_type' => $bounceType,
                    'reason' => $reason,
                ],
                null,
                null,
                null,
                $reason
            );
        }

        $subscriber = null;
        $email = (string) ($recipient->email ?? '');
        if ($email !== '') {
            $subscriber = ListSubscriber::query()
                ->when($campaign?->list_id, fn ($q) => $q->where('list_id', $campaign->list_id))
                ->where('email', strtolower(trim($email)))
                ->first();
        }

        if ($subscriber && $isHardBounce) {
            $subscriber->update([
                'status' => 'bounced',
                'is_bounced' => true,
                'bounced_at' => now(),
                'suppressed_at' => now(),
            ]);

            SuppressionList::firstOrCreate(
                [
                    'customer_id' => $subscriber->list?->customer_id,
                    'email' => $subscriber->email,
                ],
                [
                    'reason' => 'bounce',
                    'reason_description' => $reason,
                    'subscriber_id' => $subscriber->id,
                    'campaign_id' => $campaign?->id,
                    'suppressed_at' => now(),
                ]
            );
        }

        Log::info('SES bounce processed', [
            'email' => $email,
            'bounce_type' => $bounceType,
            'reason' => $reason,
            'campaign_id' => $campaign?->id,
            'recipient_id' => $recipient->id,
        ]);

        return response()->json(['status' => 'processed'], 200);
    }

    protected function resolveRecipientFromSnsBouncePayload(array $message): ?CampaignRecipient
    {
        $mail = $message['mail'] ?? [];

        $headers = $mail['headers'] ?? [];
        if (is_array($headers)) {
            foreach ($headers as $header) {
                if (!is_array($header)) {
                    continue;
                }
                $name = $header['name'] ?? null;
                $value = $header['value'] ?? null;
                if (is_string($name) && strtolower($name) === 'x-recipient-uuid' && is_string($value) && trim($value) !== '') {
                    $found = CampaignRecipient::where('uuid', trim($value))->first();
                    if ($found) {
                        return $found;
                    }
                }
            }
        }

        $destinations = $mail['destination'] ?? null;
        $email = null;
        if (is_array($destinations) && isset($destinations[0]) && is_string($destinations[0])) {
            $email = $destinations[0];
        }

        if (!is_string($email) || trim($email) === '') {
            return null;
        }

        return CampaignRecipient::where('email', trim($email))
            ->latest('id')
            ->first();
    }

    protected function handleOpen(Request $request, array $message): void
    {
        $recipient = $this->resolveRecipientFromEventPublishingPayload($message);
        if (!$recipient) {
            return;
        }

        $campaign = $recipient->campaign;
        if ($campaign && $campaign->track_opens && !$recipient->isOpened()) {
            $recipient->markAsOpened();
            $campaign->incrementOpenedCount();

            CampaignLog::logEvent(
                $campaign->id,
                'opened',
                $recipient->id,
                ['email' => $recipient->email, 'provider' => 'ses'],
                $request->ip(),
                $request->userAgent()
            );
        }
    }

    protected function handleClick(Request $request, array $message): void
    {
        $recipient = $this->resolveRecipientFromEventPublishingPayload($message);
        if (!$recipient) {
            return;
        }

        $campaign = $recipient->campaign;
        $url = data_get($message, 'click.link');

        if ($campaign && $campaign->track_clicks) {
            $wasOpened = $recipient->isOpened();
            $wasClicked = $recipient->isClicked();

            if ($campaign->track_opens && !$wasOpened) {
                $recipient->markAsOpened();
                $campaign->incrementOpenedCount();
            }

            if (!$wasClicked) {
                $recipient->markAsClicked();
                $campaign->incrementClickedCount();
            }

            CampaignLog::logEvent(
                $campaign->id,
                'clicked',
                $recipient->id,
                ['email' => $recipient->email, 'provider' => 'ses'],
                $request->ip(),
                $request->userAgent(),
                is_string($url) ? $url : null
            );
        }
    }

    protected function resolveRecipientFromEventPublishingPayload(array $message): ?CampaignRecipient
    {
        $headers = data_get($message, 'mail.headers');
        $uuid = null;

        if (is_array($headers)) {
            foreach ($headers as $header) {
                $name = $header['name'] ?? null;
                if (!is_string($name)) {
                    continue;
                }
                if (strtolower($name) === 'x-recipient-uuid') {
                    $value = $header['value'] ?? null;
                    if (is_string($value) && trim($value) !== '') {
                        $uuid = trim($value);
                        break;
                    }
                }
            }
        }

        if (is_string($uuid) && $uuid !== '') {
            $found = CampaignRecipient::where('uuid', $uuid)->first();
            if ($found) {
                return $found;
            }
        }

        $destinations = data_get($message, 'mail.destination');
        $email = null;
        if (is_array($destinations) && isset($destinations[0]) && is_string($destinations[0])) {
            $email = $destinations[0];
        }

        if (!is_string($email) || trim($email) === '') {
            return null;
        }

        return CampaignRecipient::where('email', trim($email))
            ->latest('id')
            ->first();
    }
}
