<?php

namespace App\Http\Controllers;

use App\Models\CampaignRecipient;
use App\Models\SubscriptionForm;
use App\Models\EmailList;
use App\Models\ListSubscriber;
use App\Services\ListSubscriberService;
use App\Services\EmailVerificationService;
use Illuminate\Http\Request;
use App\Notifications\NewSubscriberNotification;
use App\Notifications\SubscriberUnsubscribedNotification;

class PublicSubscriptionController extends Controller
{
    public function __construct(
        protected ListSubscriberService $listSubscriberService
    ) {}

    /**
     * Display the subscription form (embedded/popup).
     */
    public function show(string $slug)
    {
        $form = SubscriptionForm::where('slug', $slug)
            ->where('is_active', true)
            ->with('emailList')
            ->firstOrFail();

        return view('public.subscribe', compact('form'));
    }

    /**
     * Handle subscription form submission (API).
     */
    public function subscribe(Request $request, string $slug)
    {
        $form = SubscriptionForm::where('slug', $slug)
            ->where('is_active', true)
            ->with('emailList')
            ->firstOrFail();

        // Check if this is an API request (from /api endpoint)
        $isApiRequest = $request->route()->getName() === 'public.subscribe.api' || 
                       $request->is('subscribe/*/api') ||
                       $request->wantsJson() ||
                       $request->expectsJson();

        try {
            $validated = $request->validate([
                'email' => ['required', 'email', 'max:255'],
                'first_name' => ['nullable', 'string', 'max:255'],
                'last_name' => ['nullable', 'string', 'max:255'],
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            if ($isApiRequest) {
                return response()->json([
                    'message' => 'Validation failed',
                    'errors' => $e->errors(),
                    'status' => 'error'
                ], 422);
            }
            throw $e;
        }

        $normalizedEmail = strtolower(trim($validated['email']));

        // Check if subscriber already exists (including soft-deleted)
        $existing = $form->emailList->subscribers()
            ->withTrashed()
            ->where('email', $normalizedEmail)
            ->first();

        $wasCreated = false;

        if ($existing) {
            $shouldResubscribe = (method_exists($existing, 'trashed') && $existing->trashed())
                || $existing->status === 'unsubscribed';

            if ($existing->status === 'blacklisted') {
                if ($isApiRequest) {
                    return response()->json([
                        'message' => 'This email address cannot be subscribed to this list.',
                        'status' => 'blocked'
                    ], 403);
                }

                return back()->with('error', 'This email address cannot be subscribed to this list.');
            }

            if (!$shouldResubscribe) {
                if ($isApiRequest) {
                    return response()->json([
                        'message' => 'You are already subscribed to this list.',
                        'status' => 'exists'
                    ], 200);
                }

                return back()->with('error', 'You are already subscribed to this list.');
            }

            $subscriber = $this->listSubscriberService->resubscribe($form->emailList, $existing, [
                'email' => $normalizedEmail,
                'first_name' => $validated['first_name'] ?? null,
                'last_name' => $validated['last_name'] ?? null,
                'source' => 'form',
                'ip_address' => $request->ip(),
            ]);
        } else {
            // Create subscriber
            $subscriber = $this->listSubscriberService->create($form->emailList, [
                'email' => $normalizedEmail,
                'first_name' => $validated['first_name'] ?? null,
                'last_name' => $validated['last_name'] ?? null,
                'source' => 'form',
                'ip_address' => $request->ip(),
            ]);

            $wasCreated = true;
        }

        // Notify list owner (customer) about new subscriber
        $customer = $form->emailList->customer;
        if ($customer) {
            $customer->notify(new NewSubscriberNotification($form->emailList, $subscriber));
        }

        // Increment form submissions count
        $form->increment('submissions_count');

        if ($isApiRequest) {
            return response()->json([
                'message' => 'Successfully subscribed! Please check your email to confirm.',
                'status' => 'success',
                'subscriber' => [
                    'email' => $subscriber->email,
                    'status' => $subscriber->status,
                ]
            ], $wasCreated ? 201 : 200);
        }

        return back()->with('success', 'Successfully subscribed! Please check your email to confirm.');
    }

    /**
     * Confirm subscription via email token.
     */
    public function confirm(string $token)
    {
        $verificationService = app(EmailVerificationService::class);
        $verification = $verificationService->verify($token);

        if (!$verification || !$verification->subscriber) {
            return view('public.subscription-confirmed', [
                'success' => false,
                'message' => 'Invalid or expired confirmation link.'
            ]);
        }

        return view('public.subscription-confirmed', [
            'success' => true,
            'message' => 'Your subscription has been confirmed!',
            'list' => $verification->subscriber->list
        ]);
    }

    /**
     * Unsubscribe from a list.
     */
    public function unsubscribe(EmailList $list, string $email, string $token)
    {
        // Verify token
        $expectedToken = hash('sha256', $email . $list->id . config('app.key'));
        
        if (!hash_equals($expectedToken, $token)) {
            return view('public.unsubscribe', [
                'success' => false,
                'message' => 'Invalid unsubscribe link.'
            ]);
        }

        $subscriber = ListSubscriber::where('list_id', $list->id)
            ->where('email', $email)
            ->first();

        if (!$subscriber) {
            return view('public.unsubscribe', [
                'success' => false,
                'message' => 'Subscriber not found.'
            ]);
        }

        // Unsubscribe
        app(ListSubscriberService::class)->unsubscribe($subscriber);

        // Notify list owner about unsubscribe
        if ($list->customer) {
            $list->customer->notify(new SubscriberUnsubscribedNotification($list, $subscriber));
        }

        // Redirect if URL is set
        if ($list->unsubscribe_redirect_url) {
            return redirect($list->unsubscribe_redirect_url);
        }

        return view('public.unsubscribe', [
            'success' => true,
            'message' => 'You have been successfully unsubscribed.',
            'list' => $list
        ]);
    }

    /**
     * Unsubscribe from campaign using UUID.
     */
    public function unsubscribeByUuid(string $uuid)
    {
        $recipient = CampaignRecipient::where('uuid', $uuid)->firstOrFail();
        
        // Find the subscriber in the email list
        $subscriber = ListSubscriber::where('list_id', $recipient->campaign->list_id)
            ->where('email', $recipient->email)
            ->first();

        if ($subscriber) {
            app(ListSubscriberService::class)->unsubscribe($subscriber);

            // Notify list owner about unsubscribe
            $listForNotification = $subscriber->emailList;
            if ($listForNotification && $listForNotification->customer) {
                $listForNotification->customer->notify(
                    new SubscriberUnsubscribedNotification($listForNotification, $subscriber)
                );
            }
        }

        // Mark recipient as unsubscribed in campaign
        $recipient->update(['status' => 'unsubscribed']);

        $list = $recipient->campaign->emailList;
        
        // Redirect if URL is set
        if ($list && $list->unsubscribe_redirect_url) {
            return redirect($list->unsubscribe_redirect_url);
        }

        return view('public.unsubscribe', [
            'success' => true,
            'message' => 'You have been successfully unsubscribed.',
            'list' => $list
        ]);
    }
}

