<?php

namespace App\Http\Controllers\Customer;

use App\Http\Controllers\Controller;
use App\Models\EmailList;
use App\Models\SubscriptionForm;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class SubscriptionFormController extends Controller
{
    /**
     * Display a listing of forms for a list.
     */
    public function index(EmailList $list)
    {
        $forms = $list->subscriptionForms()->latest()->get();

        return view('customer.lists.forms.index', compact('list', 'forms'));
    }

    /**
     * Show the form for creating a new subscription form.
     */
    public function create(EmailList $list)
    {
        return view('customer.lists.forms.create', compact('list'));
    }

    /**
     * Store a newly created subscription form.
     */
    public function store(Request $request, EmailList $list)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'title' => ['nullable', 'string', 'max:255'],
            'type' => ['required', 'in:embedded,popup,api'],
            'description' => ['nullable', 'string'],
            'fields' => ['nullable', 'array'],
            'settings' => ['nullable', 'array'],
            'gdpr_checkbox' => ['nullable', 'boolean'],
            'gdpr_text' => ['nullable', 'string'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        $settings = $request->input('settings', []);
        $settings['show_title'] = (bool) ($settings['show_title'] ?? true);
        $settings['show_description'] = (bool) ($settings['show_description'] ?? true);

        $form = SubscriptionForm::create([
            'list_id' => $list->id,
            'name' => $validated['name'],
            'title' => $validated['title'] ?? null,
            'type' => $validated['type'],
            'slug' => Str::slug($validated['name']) . '-' . Str::random(8),
            'description' => $validated['description'] ?? null,
            'fields' => $request->input('fields', ['email', 'first_name', 'last_name']),
            'settings' => $settings,
            'gdpr_checkbox' => $validated['gdpr_checkbox'] ?? false,
            'gdpr_text' => $validated['gdpr_text'] ?? null,
            'is_active' => $validated['is_active'] ?? true,
        ]);

        return redirect()
            ->route('customer.lists.forms.show', [$list, $form])
            ->with('success', 'Subscription form created successfully.');
    }

    /**
     * Display the specified subscription form.
     */
    public function show(EmailList $list, SubscriptionForm $form)
    {
        return view('customer.lists.forms.show', compact('list', 'form'));
    }

    /**
     * Show the form for editing the specified subscription form.
     */
    public function edit(EmailList $list, SubscriptionForm $form)
    {
        return view('customer.lists.forms.edit', compact('list', 'form'));
    }

    /**
     * Update the specified subscription form.
     */
    public function update(Request $request, EmailList $list, SubscriptionForm $form)
    {
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'title' => ['nullable', 'string', 'max:255'],
            'type' => ['required', 'in:embedded,popup,api'],
            'description' => ['nullable', 'string'],
            'fields' => ['nullable', 'array'],
            'settings' => ['nullable', 'array'],
            'gdpr_checkbox' => ['nullable', 'boolean'],
            'gdpr_text' => ['nullable', 'string'],
            'is_active' => ['nullable', 'boolean'],
        ]);

        $settings = $request->input('settings', $form->settings ?? []);
        $settings['show_title'] = (bool) ($settings['show_title'] ?? true);
        $settings['show_description'] = (bool) ($settings['show_description'] ?? true);

        $updateData = $validated;
        $updateData['fields'] = $request->input('fields', $form->fields);
        $updateData['settings'] = $settings;
        $form->update($updateData);

        return redirect()
            ->route('customer.lists.forms.show', [$list, $form])
            ->with('success', 'Subscription form updated successfully.');
    }

    /**
     * Remove the specified subscription form.
     */
    public function destroy(EmailList $list, SubscriptionForm $form)
    {
        $form->delete();

        return redirect()
            ->route('customer.lists.forms.index', $list)
            ->with('success', 'Subscription form deleted successfully.');
    }
}

