<?php

namespace App\Http\Controllers\Customer;

use App\Http\Controllers\Controller;
use App\Models\EmailList;
use App\Models\ListSubscriber;
use App\Models\SubscriberImport;
use App\Services\ListSubscriberService;
use App\Services\SubscriberImportProcessor;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;

class ListSubscriberController extends Controller
{
    public function __construct(
        protected ListSubscriberService $listSubscriberService
    ) {}

    /**
     * Display a listing of subscribers for a list.
     */
    public function index(Request $request, EmailList $list)
    {
        $filters = $request->only(['search', 'status']);
        $subscribers = $this->listSubscriberService->getPaginated($list, $filters);

        return view('customer.lists.subscribers.index', compact('list', 'subscribers', 'filters'));
    }

    /**
     * Show the form for creating a new subscriber.
     */
    public function create(EmailList $list)
    {
        return view('customer.lists.subscribers.create', compact('list'));
    }

    /**
     * Store a newly created subscriber.
     */
    public function store(Request $request, EmailList $list)
    {
        $validated = $request->validate([
            'email' => ['required', 'email', 'max:255'],
            'first_name' => ['nullable', 'string', 'max:255'],
            'last_name' => ['nullable', 'string', 'max:255'],
            'source' => ['nullable', 'string', 'max:255'],
            'tags' => ['nullable', 'array'],
        ]);

        $subscriber = $this->listSubscriberService->create($list, $validated);

        return redirect()
            ->route('customer.lists.subscribers.index', $list)
            ->with('success', 'Subscriber added successfully.');
    }

    /**
     * Display the specified subscriber.
     */
    public function show(EmailList $list, ListSubscriber $subscriber)
    {
        return view('customer.lists.subscribers.show', compact('list', 'subscriber'));
    }

    /**
     * Show the form for editing the specified subscriber.
     */
    public function edit(EmailList $list, ListSubscriber $subscriber)
    {
        return view('customer.lists.subscribers.edit', compact('list', 'subscriber'));
    }

    /**
     * Update the specified subscriber.
     */
    public function update(Request $request, EmailList $list, ListSubscriber $subscriber)
    {
        $validated = $request->validate([
            'email' => ['required', 'email', 'max:255'],
            'first_name' => ['nullable', 'string', 'max:255'],
            'last_name' => ['nullable', 'string', 'max:255'],
            'status' => ['nullable', 'in:confirmed,unconfirmed,unsubscribed,blacklisted,bounced'],
            'tags' => ['nullable', 'array'],
            'notes' => ['nullable', 'string'],
        ]);

        $this->listSubscriberService->update($subscriber, $validated);

        return redirect()
            ->route('customer.lists.subscribers.show', [$list, $subscriber])
            ->with('success', 'Subscriber updated successfully.');
    }

    /**
     * Remove the specified subscriber.
     */
    public function destroy(EmailList $list, ListSubscriber $subscriber)
    {
        $this->listSubscriberService->delete($subscriber);

        return redirect()
            ->route('customer.lists.subscribers.index', $list)
            ->with('success', 'Subscriber deleted successfully.');
    }

    /**
     * Show CSV import form.
     */
    public function showImport(EmailList $list)
    {
        return view('customer.lists.subscribers.import', compact('list'));
    }

    /**
     * Handle CSV import.
     */
    public function import(Request $request, EmailList $list)
    {
        $validated = $request->validate([
            'csv_file' => ['required', 'file', 'mimes:csv,txt', 'max:51200'],
            'column_mapping' => ['required', 'array'],
            'column_mapping.email' => ['required', 'string'],
            'column_mapping.first_name' => ['nullable', 'string'],
            'column_mapping.last_name' => ['nullable', 'string'],
            'skip_duplicates' => ['nullable', 'boolean'],
            'update_existing' => ['nullable', 'boolean'],
        ]);

        $file = $request->file('csv_file');
        
        if (!$file || !$file->isValid()) {
            return redirect()
                ->route('customer.lists.subscribers.import', $list)
                ->with('error', 'Invalid file upload. Please try again.')
                ->withInput();
        }
        
        // Ensure imports directory exists using Storage facade
        try {
            if (!Storage::disk('local')->exists('imports')) {
                Storage::disk('local')->makeDirectory('imports');
            }
        } catch (\Exception $e) {
            Log::error("Failed to create imports directory: " . $e->getMessage());
            return redirect()
                ->route('customer.lists.subscribers.import', $list)
                ->with('error', 'Failed to create import directory. Please check storage permissions.')
                ->withInput();
        }
        
        // Store file and get full path
        $fileName = 'subscribers_' . time() . '_' . uniqid() . '.csv';
        
        try {
            // Use Storage facade for more reliable file handling
            $filePath = Storage::disk('local')->putFileAs('imports', $file, $fileName);
            
            if ($filePath === false) {
                Log::error("Storage::putFileAs() returned false for file: {$fileName}");
                return redirect()
                    ->route('customer.lists.subscribers.import', $list)
                    ->with('error', 'Failed to save uploaded file. Please try again.')
                    ->withInput();
            }
            
            $fullPath = Storage::disk('local')->path($filePath);

            // Verify file was saved
            if (!Storage::disk('local')->exists($filePath)) {
                Log::error("File not found after storage: {$filePath}");
                return redirect()
                    ->route('customer.lists.subscribers.import', $list)
                    ->with('error', 'File was not saved correctly. Please try again.')
                    ->withInput();
            }
            
            // Verify file is readable
            if (!is_readable($fullPath)) {
                Log::error("File is not readable: {$fullPath}");
                return redirect()
                    ->route('customer.lists.subscribers.import', $list)
                    ->with('error', 'File is not readable. Please check permissions.')
                    ->withInput();
            }
            
            Log::info("File successfully stored: {$fullPath}");
            
        } catch (\Exception $e) {
            Log::error("Exception during file storage: " . $e->getMessage(), [
                'file' => $fileName,
                'trace' => $e->getTraceAsString()
            ]);
            return redirect()
                ->route('customer.lists.subscribers.import', $list)
                ->with('error', 'Error saving file: ' . $e->getMessage())
                ->withInput();
        }

        $customerId = (int) ($list->customer_id ?? 0);

        $subscriberImport = SubscriberImport::create([
            'customer_id' => $customerId,
            'list_id' => (int) $list->id,
            'status' => 'queued',
            'source' => 'csv_import',
            'ip_address' => $request->ip(),
            'stored_path' => $filePath,
            'total_rows' => 0,
            'processed_count' => 0,
            'imported_count' => 0,
            'updated_count' => 0,
            'skipped_count' => 0,
            'error_count' => 0,
        ]);

        \App\Jobs\ImportSubscribersJob::dispatch(
            $list,
            $fullPath,
            $validated['column_mapping'],
            $validated['skip_duplicates'] ?? true,
            $validated['update_existing'] ?? false,
            'csv_import',
            $request->ip(),
            (int) $subscriberImport->id
        )->onQueue('imports');

        return redirect()
            ->route('customer.lists.subscribers.index', $list)
            ->with('success', 'Import started. You will be notified when it completes.');
    }

    public function importAjaxStart(Request $request, EmailList $list)
    {
        $customerId = auth('customer')->id();
        if (!$customerId || (int) $list->customer_id !== (int) $customerId) {
            abort(404);
        }

        $validated = $request->validate([
            'csv_file' => ['required', 'file', 'mimes:csv,txt', 'max:51200'],
            'column_mapping' => ['required', 'array'],
            'column_mapping.email' => ['required', 'string'],
            'column_mapping.first_name' => ['nullable', 'string'],
            'column_mapping.last_name' => ['nullable', 'string'],
            'skip_duplicates' => ['nullable', 'boolean'],
            'update_existing' => ['nullable', 'boolean'],
        ]);

        $file = $request->file('csv_file');
        if (!$file || !$file->isValid()) {
            return response()->json([
                'success' => false,
                'message' => 'Invalid file upload.',
            ], 422);
        }

        try {
            if (!Storage::disk('local')->exists('imports')) {
                Storage::disk('local')->makeDirectory('imports');
            }
        } catch (\Throwable $e) {
            Log::error('Failed to create imports directory', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to create import directory. Please check storage permissions.',
            ], 500);
        }

        $fileName = 'subscribers_' . time() . '_' . uniqid() . '.csv';

        try {
            $filePath = Storage::disk('local')->putFileAs('imports', $file, $fileName);
            if ($filePath === false) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to save uploaded file. Please try again.',
                ], 500);
            }

            if (!Storage::disk('local')->exists($filePath)) {
                return response()->json([
                    'success' => false,
                    'message' => 'Upload completed but the file could not be found on the server. Please check storage permissions.',
                ], 500);
            }
        } catch (\Throwable $e) {
            Log::error('Exception during file storage', ['error' => $e->getMessage()]);
            return response()->json([
                'success' => false,
                'message' => 'Error saving file: ' . $e->getMessage(),
            ], 500);
        }

        $subscriberImport = SubscriberImport::create([
            'customer_id' => (int) $customerId,
            'list_id' => (int) $list->id,
            'status' => 'queued',
            'source' => 'csv_import',
            'ip_address' => $request->ip(),
            'stored_path' => $filePath,
            'column_mapping' => $validated['column_mapping'],
            'skip_duplicates' => (bool) ($validated['skip_duplicates'] ?? true),
            'update_existing' => (bool) ($validated['update_existing'] ?? false),
            'file_offset' => 0,
            'total_rows' => 0,
            'processed_count' => 0,
            'imported_count' => 0,
            'updated_count' => 0,
            'skipped_count' => 0,
            'error_count' => 0,
        ]);

        try {
            app(SubscriberImportProcessor::class)->processImportNow($subscriberImport, 400, 6);
        } catch (\Throwable $e) {
            // Processor will mark failed itself; just continue.
        }

        return response()->json([
            'success' => true,
            'import_id' => (int) $subscriberImport->id,
        ]);
    }

    public function importAjaxStep(Request $request, EmailList $list)
    {
        $customerId = auth('customer')->id();
        if (!$customerId || (int) $list->customer_id !== (int) $customerId) {
            abort(404);
        }

        $validated = $request->validate([
            'import_id' => ['required', 'integer'],
        ]);

        $import = SubscriberImport::query()
            ->whereKey((int) $validated['import_id'])
            ->where('customer_id', (int) $customerId)
            ->where('list_id', (int) $list->id)
            ->firstOrFail();

        if (in_array((string) $import->status, ['completed', 'failed'], true)) {
            return $this->importStats($request, $list);
        }

        try {
            app(SubscriberImportProcessor::class)->processImportNow($import, 800, 6);
        } catch (\Throwable $e) {
            // Processor handles marking failed.
        }

        return $this->importStats($request, $list);
    }

    public function importStats(Request $request, EmailList $list)
    {
        $customerId = auth('customer')->id();
        if (!$customerId || (int) $list->customer_id !== (int) $customerId) {
            abort(404);
        }

        $import = SubscriberImport::query()
            ->where('customer_id', (int) $customerId)
            ->where('list_id', (int) $list->id)
            ->latest()
            ->first();

        if (!$import) {
            return response()->json([
                'success' => true,
                'import' => null,
            ]);
        }

        $total = max(0, (int) $import->total_rows);
        $processed = max(0, (int) $import->processed_count);
        $percent = $total > 0 ? round(($processed / $total) * 100, 1) : 0;

        if ((string) $import->status === 'failed'
            && (string) ($import->failure_reason ?? '') === 'Import file not found.'
            && $total > 0
            && $processed >= $total
        ) {
            $import->update([
                'status' => 'completed',
                'failure_reason' => null,
            ]);

            $import = $import->fresh();
        }

        return response()->json([
            'success' => true,
            'import' => [
                'id' => (int) $import->id,
                'status' => (string) $import->status,
                'total_rows' => $total,
                'processed_count' => $processed,
                'imported_count' => (int) $import->imported_count,
                'updated_count' => (int) $import->updated_count,
                'skipped_count' => (int) $import->skipped_count,
                'error_count' => (int) $import->error_count,
                'percent' => $percent,
                'failure_reason' => $import->failure_reason,
                'started_at' => $import->started_at?->toIso8601String(),
                'finished_at' => $import->finished_at?->toIso8601String(),
            ],
        ]);
    }

    /**
     * Confirm a subscriber.
     */
    public function confirm(EmailList $list, ListSubscriber $subscriber)
    {
        $this->listSubscriberService->confirm($subscriber);

        return redirect()
            ->route('customer.lists.subscribers.show', [$list, $subscriber])
            ->with('success', 'Subscriber confirmed successfully.');
    }

    /**
     * Unsubscribe a subscriber.
     */
    public function unsubscribe(EmailList $list, ListSubscriber $subscriber)
    {
        $this->listSubscriberService->unsubscribe($subscriber);

        return redirect()
            ->route('customer.lists.subscribers.show', [$list, $subscriber])
            ->with('success', 'Subscriber unsubscribed successfully.');
    }

    /**
     * Resend confirmation email to subscriber.
     */
    public function resendConfirmation(EmailList $list, ListSubscriber $subscriber)
    {
        try {
            $this->listSubscriberService->resendConfirmationEmail($subscriber);

            return redirect()
                ->route('customer.lists.subscribers.show', [$list, $subscriber])
                ->with('success', 'Confirmation email sent successfully.');
        } catch (\Exception $e) {
            return redirect()
                ->route('customer.lists.subscribers.show', [$list, $subscriber])
                ->with('error', $e->getMessage());
        }
    }

    /**
     * Bulk delete subscribers.
     */
    public function bulkDelete(Request $request, EmailList $list)
    {
        $validated = $request->validate([
            'subscriber_ids' => ['required', 'string'],
        ]);

        $ids = explode(',', $validated['subscriber_ids']);
        $ids = array_filter(array_map('intval', $ids));

        if (empty($ids)) {
            return redirect()
                ->route('customer.lists.subscribers.index', $list)
                ->with('error', 'No subscribers selected.');
        }

        $count = ListSubscriber::where('list_id', $list->id)
            ->whereIn('id', $ids)
            ->delete();

        return redirect()
            ->route('customer.lists.subscribers.index', $list)
            ->with('success', "Successfully deleted {$count} subscriber(s).");
    }

    /**
     * Bulk confirm subscribers.
     */
    public function bulkConfirm(Request $request, EmailList $list)
    {
        $validated = $request->validate([
            'subscriber_ids' => ['required', 'string'],
        ]);

        $ids = explode(',', $validated['subscriber_ids']);
        $ids = array_filter(array_map('intval', $ids));

        if (empty($ids)) {
            return redirect()
                ->route('customer.lists.subscribers.index', $list)
                ->with('error', 'No subscribers selected.');
        }

        $count = ListSubscriber::where('list_id', $list->id)
            ->whereIn('id', $ids)
            ->where('status', 'unconfirmed')
            ->update([
                'status' => 'confirmed',
                'confirmed_at' => now(),
            ]);

        return redirect()
            ->route('customer.lists.subscribers.index', $list)
            ->with('success', "Successfully confirmed {$count} subscriber(s).");
    }

    /**
     * Bulk unsubscribe subscribers.
     */
    public function bulkUnsubscribe(Request $request, EmailList $list)
    {
        $validated = $request->validate([
            'subscriber_ids' => ['required', 'string'],
        ]);

        $ids = explode(',', $validated['subscriber_ids']);
        $ids = array_filter(array_map('intval', $ids));

        if (empty($ids)) {
            return redirect()
                ->route('customer.lists.subscribers.index', $list)
                ->with('error', 'No subscribers selected.');
        }

        $subscribers = ListSubscriber::where('list_id', $list->id)
            ->whereIn('id', $ids)
            ->get();

        $count = 0;
        foreach ($subscribers as $subscriber) {
            $this->listSubscriberService->unsubscribe($subscriber);
            $count++;
        }

        return redirect()
            ->route('customer.lists.subscribers.index', $list)
            ->with('success', "Successfully unsubscribed {$count} subscriber(s).");
    }

    /**
     * Bulk resend confirmation emails.
     */
    public function bulkResend(Request $request, EmailList $list)
    {
        $validated = $request->validate([
            'subscriber_ids' => ['required', 'string'],
        ]);

        $ids = explode(',', $validated['subscriber_ids']);
        $ids = array_filter(array_map('intval', $ids));

        if (empty($ids)) {
            return redirect()
                ->route('customer.lists.subscribers.index', $list)
                ->with('error', 'No subscribers selected.');
        }

        $subscribers = ListSubscriber::where('list_id', $list->id)
            ->whereIn('id', $ids)
            ->where('status', 'unconfirmed')
            ->get();

        $count = 0;
        $errors = 0;

        foreach ($subscribers as $subscriber) {
            try {
                $this->listSubscriberService->resendConfirmationEmail($subscriber);
                $count++;
            } catch (\Exception $e) {
                Log::error("Failed to resend confirmation to subscriber {$subscriber->id}: " . $e->getMessage());
                $errors++;
            }
        }

        $message = "Successfully sent confirmation emails to {$count} subscriber(s).";
        if ($errors > 0) {
            $message .= " {$errors} email(s) failed to send.";
        }

        return redirect()
            ->route('customer.lists.subscribers.index', $list)
            ->with($errors > 0 ? 'warning' : 'success', $message);
    }
}
