<?php

namespace App\Http\Controllers\Customer;

use App\Http\Controllers\Controller;
use App\Models\DeliveryServer;
use App\Models\GoogleIntegration;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Schema;

class IntegrationController extends Controller
{
    public function __construct()
    {
        $this->middleware('customer.access:servers.permissions.can_access_delivery_servers')->only(['index']);
    }

    public function index(Request $request)
    {
        $customer = auth('customer')->user();
        $canAccessGoogle = $customer ? (bool) $customer->groupAllows('integrations.permissions.can_access_google') : false;

        $allowedTabs = ['delivery-servers', 'wordpress'];
        if ($canAccessGoogle) {
            $allowedTabs[] = 'google';
        }

        $tab = (string) $request->query('tab', 'delivery-servers');
        if (!in_array($tab, $allowedTabs, true)) {
            $tab = 'delivery-servers';
        }

        $providers = [
            [
                'key' => 'mailgun',
                'label' => 'Mailgun',
                'type' => 'mailgun',
                'description' => 'Send emails using Mailgun API and webhooks.',
                'supported' => true,
            ],
            [
                'key' => 'sendgrid',
                'label' => 'SendGrid',
                'type' => 'sendgrid',
                'description' => 'Send emails with SendGrid API and event webhooks.',
                'supported' => true,
            ],
            [
                'key' => 'postmark',
                'label' => 'Postmark',
                'type' => 'postmark',
                'description' => 'Deliver transactional emails through Postmark.',
                'supported' => true,
            ],
            [
                'key' => 'sparkpost',
                'label' => 'SparkPost',
                'type' => 'sparkpost',
                'description' => 'Deliver emails using SparkPost API.',
                'supported' => true,
            ],
            [
                'key' => 'amazon-ses',
                'label' => 'Amazon SES',
                'type' => 'amazon-ses',
                'description' => 'Send emails through Amazon SES (API).',
                'supported' => true,
            ],
            [
                'key' => 'zeptomail-api',
                'label' => 'ZeptoMail API',
                'type' => 'zeptomail-api',
                'description' => 'Send emails via ZeptoMail Email API (templates or raw HTML/text).',
                'supported' => true,
            ],
        ];

        $mustAddOwn = $customer ? (bool) $customer->groupSetting('servers.permissions.must_add_delivery_server', false) : false;
        $canUseSystem = $customer ? (bool) $customer->groupSetting('servers.permissions.can_use_system_servers', false) : false;

        $supportedTypes = collect($providers)
            ->filter(fn ($p) => (bool) ($p['supported'] ?? false))
            ->pluck('type')
            ->values()
            ->all();

        $configured = DeliveryServer::query()
            ->when($customer, function ($q) use ($customer, $mustAddOwn, $canUseSystem) {
                $q->when($mustAddOwn, function ($inner) use ($customer) {
                    $inner->where('customer_id', $customer->id);
                }, function ($inner) use ($customer, $canUseSystem) {
                    $inner->where(function ($sub) use ($customer, $canUseSystem) {
                        $sub->where('customer_id', $customer->id);
                        if ($canUseSystem) {
                            $sub->orWhere(function ($sys) {
                                $sys->whereNull('customer_id')->where('status', 'active');
                            });
                        }
                    });
                });
            }, function ($q) {
                $q->whereRaw('1 = 0');
            })
            ->whereIn('type', $supportedTypes)
            ->selectRaw('type, COUNT(*) as cnt')
            ->groupBy('type')
            ->pluck('cnt', 'type')
            ->map(fn ($v) => ((int) $v) > 0)
            ->toArray();

        $serversByType = DeliveryServer::query()
            ->when($customer, fn ($q) => $q->where('customer_id', $customer->id), fn ($q) => $q->whereRaw('1 = 0'))
            ->whereIn('type', $supportedTypes)
            ->orderByDesc('id')
            ->get()
            ->groupBy('type')
            ->map(fn ($items) => $items->first())
            ->all();

        $googleIntegrations = collect();
        if ($customer && $canAccessGoogle && Schema::hasTable('google_integrations')) {
            $googleIntegrations = GoogleIntegration::query()
                ->where('customer_id', $customer->id)
                ->whereIn('service', ['sheets', 'drive'])
                ->get()
                ->keyBy('service');
        }

        return view('customer.integrations.index', compact('tab', 'providers', 'configured', 'serversByType', 'canAccessGoogle', 'googleIntegrations'));
    }

    public function downloadWordpressPlugin(Request $request)
    {
        $path = base_path('wordpress-plugin/mailpurse-integration.zip');

        abort_unless(is_file($path), 404);

        return response()->download($path, 'mailpurse-integration.zip');
    }
}
