<?php

namespace App\Http\Controllers\Customer;

use App\Http\Controllers\Controller;
use App\Jobs\StartEmailValidationRunJob;
use App\Models\EmailList;
use App\Models\EmailValidationRun;
use App\Models\EmailValidationTool;
use App\Services\Billing\UsageService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class EmailValidationRunController extends Controller
{
    private const META_ALLOWED_GROUP_IDS_KEY = 'allowed_customer_group_ids';

    protected function customerGroupIds($customer): array
    {
        if (!$customer) {
            return [];
        }

        $groups = $customer->relationLoaded('customerGroups')
            ? $customer->customerGroups
            : $customer->customerGroups()->get();

        return $groups
            ->pluck('id')
            ->map(fn ($id) => (int) $id)
            ->filter(fn ($id) => $id > 0)
            ->values()
            ->all();
    }

    protected function globalToolAllowedForCustomer(EmailValidationTool $tool, $customer): bool
    {
        if ($tool->customer_id !== null) {
            return false;
        }

        $allowedGroupIds = (array) data_get($tool->meta ?? [], self::META_ALLOWED_GROUP_IDS_KEY, []);
        $allowedGroupIds = array_values(array_unique(array_filter(array_map('intval', $allowedGroupIds), fn ($id) => $id > 0)));

        if (empty($allowedGroupIds)) {
            return true;
        }

        $customerGroupIds = $this->customerGroupIds($customer);
        if (empty($customerGroupIds)) {
            return false;
        }

        return count(array_intersect($allowedGroupIds, $customerGroupIds)) > 0;
    }
    protected function authorizeRun(EmailValidationRun $run): EmailValidationRun
    {
        $customerId = auth('customer')->id();

        if (!$customerId || (int) $run->customer_id !== (int) $customerId) {
            abort(404);
        }

        return $run;
    }

    protected function authorizeToolView(EmailValidationTool $tool): EmailValidationTool
    {
        $customer = auth('customer')->user();
        if (!$customer) {
            abort(404);
        }

        if ((int) $tool->customer_id === (int) $customer->id) {
            return $tool;
        }

        $mustAddOwn = (bool) $customer->groupSetting('email_validation.must_add', false);

        if (!$mustAddOwn && $tool->customer_id === null && $this->globalToolAllowedForCustomer($tool, $customer)) {
            return $tool;
        }

        abort(404);
    }

    public function index()
    {
        $customer = auth('customer')->user();

        $runs = EmailValidationRun::query()
            ->where('customer_id', $customer->id)
            ->with(['list', 'tool'])
            ->latest()
            ->paginate(15);

        return view('customer.email-validation.runs.index', compact('runs'));
    }

    public function create()
    {
        $customer = auth('customer')->user();

        $lists = EmailList::query()
            ->where('customer_id', $customer->id)
            ->orderBy('name')
            ->get();

        if ($lists->isEmpty()) {
            return redirect()
                ->route('customer.lists.index')
                ->with('error', 'Please create an email list first.');
        }

        $mustAddOwn = (bool) $customer->groupSetting('email_validation.must_add', false);

        $customerGroupIds = $this->customerGroupIds($customer);

        $tools = EmailValidationTool::query()
            ->when($mustAddOwn, function ($q) use ($customer) {
                $q->where('customer_id', $customer->id);
            }, function ($q) use ($customer) {
                $q->where(function ($sub) use ($customer) {
                    $sub->where('customer_id', $customer->id)
                        ->orWhereNull('customer_id');
                });
            })
            ->when(!$mustAddOwn, function ($query) use ($customerGroupIds) {
                $query->where(function ($sub) use ($customerGroupIds) {
                    $sub->whereNotNull('customer_id')
                        ->orWhere(function ($global) use ($customerGroupIds) {
                            $global->whereNull('customer_id')
                                ->where(function ($scope) use ($customerGroupIds) {
                                    $scope->whereNull('meta')
                                        ->orWhereNull('meta->' . self::META_ALLOWED_GROUP_IDS_KEY)
                                        ->orWhereJsonLength('meta->' . self::META_ALLOWED_GROUP_IDS_KEY, 0);

                                    if (!empty($customerGroupIds)) {
                                        $scope->orWhere(function ($allowed) use ($customerGroupIds) {
                                            foreach ($customerGroupIds as $groupId) {
                                                $allowed->orWhereJsonContains('meta->' . self::META_ALLOWED_GROUP_IDS_KEY, (int) $groupId);
                                            }
                                        });
                                    }
                                });
                        });
                });
            })
            ->where('active', true)
            ->orderBy('name')
            ->get();

        if ($tools->isEmpty()) {
            return redirect()
                ->route('customer.email-validation.tools.create')
                ->with('error', 'Please add an email validation tool first.');
        }

        return view('customer.email-validation.runs.create', compact('lists', 'tools'));
    }

    public function store(Request $request)
    {
        $customer = auth('customer')->user();

        $validated = $request->validate([
            'list_id' => ['required', 'integer'],
            'tool_id' => ['required', 'integer'],
            'invalid_action' => ['required', 'in:unsubscribe,mark_spam,delete'],
        ]);

        $list = EmailList::query()
            ->where('customer_id', $customer->id)
            ->findOrFail($validated['list_id']);

        $tool = EmailValidationTool::query()->findOrFail($validated['tool_id']);
        $this->authorizeToolView($tool);

        $total = DB::table('list_subscribers')
            ->where('list_id', $list->id)
            ->whereNull('deleted_at')
            ->whereNotNull('email')
            ->where('email', '!=', '')
            ->distinct()
            ->count('email');

        $monthlyLimit = (int) $customer->groupSetting('email_validation.monthly_limit', 0);
        if ($monthlyLimit > 0) {
            $usage = app(UsageService::class)->getUsage($customer);
            $current = (int) ($usage['email_validation_emails_this_month'] ?? 0);
            $remaining = $monthlyLimit - $current;

            if ($remaining < $total) {
                return back()->with('error', "Monthly validation limit exceeded. Remaining: {$remaining}.");
            }
        }

        $run = EmailValidationRun::create([
            'customer_id' => $customer->id,
            'list_id' => $list->id,
            'tool_id' => $tool->id,
            'status' => 'pending',
            'invalid_action' => $validated['invalid_action'],
            'total_emails' => $total,
            'processed_count' => 0,
            'deliverable_count' => 0,
            'undeliverable_count' => 0,
            'accept_all_count' => 0,
            'unknown_count' => 0,
            'error_count' => 0,
        ]);

        StartEmailValidationRunJob::dispatch($run)
            ->onQueue('email-validation');

        return redirect()
            ->route('customer.email-validation.runs.show', $run)
            ->with('success', 'Validation run created.');
    }

    public function show(EmailValidationRun $run)
    {
        $this->authorizeRun($run);

        $run->load(['list', 'tool']);

        return view('customer.email-validation.runs.show', compact('run'));
    }

    public function stats(EmailValidationRun $run)
    {
        $this->authorizeRun($run);

        $run->refresh();

        $total = max(0, (int) $run->total_emails);
        $processed = max(0, (int) $run->processed_count);
        $percent = $total > 0 ? round(($processed / $total) * 100, 1) : 0;

        return response()->json([
            'success' => true,
            'stats' => [
                'status' => $run->status,
                'total_emails' => $total,
                'processed_count' => $processed,
                'percent' => $percent,
                'deliverable_count' => (int) $run->deliverable_count,
                'undeliverable_count' => (int) $run->undeliverable_count,
                'accept_all_count' => (int) $run->accept_all_count,
                'unknown_count' => (int) $run->unknown_count,
                'error_count' => (int) $run->error_count,
            ],
        ]);
    }
}
