<?php

namespace App\Http\Controllers\Customer;

use App\Http\Controllers\Controller;
use App\Models\EmailList;
use App\Services\EmailListService;
use Illuminate\Http\Request;

class EmailListController extends Controller
{
    public function __construct(
        protected EmailListService $emailListService
    ) {
        $this->middleware('customer.access:lists.permissions.can_access_lists')->only(['index', 'show']);
        $this->middleware('customer.access:lists.permissions.can_create_lists')->only(['create', 'store']);
        $this->middleware('customer.access:lists.permissions.can_edit_lists')->only(['edit', 'update']);
        $this->middleware('customer.access:lists.permissions.can_delete_lists')->only(['destroy']);
    }

    protected function authorizeOwnership(EmailList $list): EmailList
    {
        $customerId = auth('customer')->id();

        if (!$customerId || (int) $list->customer_id !== (int) $customerId) {
            abort(404);
        }

        return $list;
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $filters = $request->only(['search', 'status']);
        $emailLists = $this->emailListService->getPaginated(auth('customer')->user(), $filters);

        return view('customer.lists.index', compact('emailLists', 'filters'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        return view('customer.lists.create');
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $customer = auth('customer')->user();
        $customer->enforceGroupLimit('lists.limits.max_lists', $customer->emailLists()->count(), 'Email list limit reached.');

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'display_name' => ['nullable', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'from_name' => ['nullable', 'string', 'max:255'],
            'from_email' => ['nullable', 'email', 'max:255'],
            'reply_to' => ['nullable', 'email', 'max:255'],
            'status' => ['nullable', 'in:active,inactive,pending'],
            'opt_in' => ['nullable', 'in:single,double'],
            'opt_out' => ['nullable', 'in:single,double'],
            'double_opt_in' => ['nullable', 'boolean'],
            'default_subject' => ['nullable', 'string', 'max:255'],
            'company_name' => ['nullable', 'string', 'max:255'],
            'company_address' => ['nullable', 'string'],
            'footer_text' => ['nullable', 'string'],
            'welcome_email_enabled' => ['nullable', 'boolean'],
            'welcome_email_subject' => ['nullable', 'string', 'max:255'],
            'welcome_email_content' => ['nullable', 'string'],
            'unsubscribe_email_enabled' => ['nullable', 'boolean'],
            'unsubscribe_email_subject' => ['nullable', 'string', 'max:255'],
            'unsubscribe_email_content' => ['nullable', 'string'],
        ]);

        $emailList = $this->emailListService->create($customer, $validated);

        return redirect()
            ->route('customer.lists.show', $emailList)
            ->with('success', 'Email list created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show(EmailList $list)
    {
        $this->authorizeOwnership($list);
        $list->load(['subscribers' => function ($query) {
            $query->latest()->limit(10);
        }]);
        
        return view('customer.lists.show', compact('list'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(EmailList $list)
    {
        $this->authorizeOwnership($list);
        return view('customer.lists.edit', compact('list'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, EmailList $list)
    {
        $this->authorizeOwnership($list);
        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'display_name' => ['nullable', 'string', 'max:255'],
            'description' => ['nullable', 'string'],
            'from_name' => ['nullable', 'string', 'max:255'],
            'from_email' => ['nullable', 'email', 'max:255'],
            'reply_to' => ['nullable', 'email', 'max:255'],
            'status' => ['nullable', 'in:active,inactive,pending'],
            'opt_in' => ['nullable', 'in:single,double'],
            'opt_out' => ['nullable', 'in:single,double'],
            'double_opt_in' => ['nullable', 'boolean'],
            'default_subject' => ['nullable', 'string', 'max:255'],
            'company_name' => ['nullable', 'string', 'max:255'],
            'company_address' => ['nullable', 'string'],
            'footer_text' => ['nullable', 'string'],
            'welcome_email_enabled' => ['nullable', 'boolean'],
            'welcome_email_subject' => ['nullable', 'string', 'max:255'],
            'welcome_email_content' => ['nullable', 'string'],
            'unsubscribe_email_enabled' => ['nullable', 'boolean'],
            'unsubscribe_email_subject' => ['nullable', 'string', 'max:255'],
            'unsubscribe_email_content' => ['nullable', 'string'],
        ]);

        $this->emailListService->update($list, $validated);

        return redirect()
            ->route('customer.lists.show', $list)
            ->with('success', 'Email list updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(EmailList $list)
    {
        $this->authorizeOwnership($list);
        $this->emailListService->delete($list);

        return redirect()
            ->route('customer.lists.index')
            ->with('success', 'Email list deleted successfully.');
    }
}
