<?php

namespace App\Http\Controllers;

use App\Models\Customer;
use App\Models\CustomerGroup;
use App\Models\Setting;
use App\Models\User;
use App\Services\AffiliateService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;
use Illuminate\Validation\ValidationException;

class AuthController extends Controller
{
    /**
     * Show the login form.
     */
    public function showLoginForm()
    {
        return view('auth.login');
    }

    /**
     * Show the registration form.
     */
    public function showRegisterForm()
    {
        return view('auth.register');
    }

    /**
     * Handle a login request.
     */
    public function login(Request $request)
    {
        $request->validate([
            'email' => ['required', 'email'],
            'password' => ['required'],
        ]);

        // Try admin login first
        if (Auth::guard('admin')->attempt($request->only('email', 'password'), $request->boolean('remember'))) {
            $request->session()->regenerate();
            /** @var User|null $user */
            $user = Auth::guard('admin')->user();
            $fallback = $this->defaultAdminRedirect($user);

            return redirect()->intended($fallback);
        }

        // Try customer login
        if (Auth::guard('customer')->attempt($request->only('email', 'password'), $request->boolean('remember'))) {
            $request->session()->regenerate();
            return redirect()->intended(route('customer.dashboard'));
        }

        throw ValidationException::withMessages([
            'email' => __('The provided credentials do not match our records.'),
        ]);
    }

    /**
     * Handle a registration request.
     */
    public function register(Request $request)
    {
        $validated = $request->validate([
            'email' => ['required', 'email', 'max:255', 'unique:customers,email'],
            'password' => ['required', 'confirmed', Password::defaults()],
        ]);

        $customer = Customer::create([
            'first_name' => 'Customer',
            'last_name' => 'User',
            'email' => $validated['email'],
            'password' => Hash::make($validated['password']),
            'status' => 'active',
        ]);

        app(AffiliateService::class)->attributeCustomerFromRequest($customer, $request);

        $newRegisteredGroupId = Setting::get('new_registered_customer_group_id');
        $fallbackGroupId = Setting::get('default_customer_group_id');
        $groupId = $newRegisteredGroupId ?: $fallbackGroupId;

        if ($groupId && CustomerGroup::query()->whereKey((int) $groupId)->exists()) {
            $customer->customerGroups()->syncWithoutDetaching([(int) $groupId]);
        }

        Auth::guard('customer')->login($customer);

        $requireVerification = (bool) Setting::get('new_register_requires_email_verification', true);

        if ($requireVerification) {
            $customer->sendEmailVerificationNotification();
            return redirect()->route('verification.notice');
        }

        $customer->markEmailAsVerified();
        return redirect()->route('customer.dashboard');
    }

    /**
     * Handle a logout request.
     */
    public function logout(Request $request)
    {
        // Logout from both guards
        Auth::guard('admin')->logout();
        Auth::guard('customer')->logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('login');
    }

    private function defaultAdminRedirect(?User $user): string
    {
        if (!$user) {
            return route('admin.login');
        }

        if (method_exists($user, 'isSuperAdmin') && $user->isSuperAdmin()) {
            return route('admin.dashboard');
        }

        if (!method_exists($user, 'hasAdminAbility')) {
            return route('admin.login');
        }

        $candidates = [
            ['ability' => 'admin.dashboard.access', 'route' => 'admin.dashboard'],
            ['ability' => 'admin.customers.access', 'route' => 'admin.customers.index'],
            ['ability' => 'admin.campaigns.access', 'route' => 'admin.campaigns.index'],
            ['ability' => 'admin.delivery_servers.access', 'route' => 'admin.delivery-servers.index'],
            ['ability' => 'admin.bounce_servers.access', 'route' => 'admin.bounce-servers.index'],
            ['ability' => 'admin.reply_servers.access', 'route' => 'admin.reply-servers.index'],
            ['ability' => 'admin.lists.access', 'route' => 'admin.lists.index'],
            ['ability' => 'admin.customer_groups.access', 'route' => 'admin.customer-groups.index'],
            ['ability' => 'admin.users.access', 'route' => 'admin.users.index'],
            ['ability' => 'admin.settings.access', 'route' => 'admin.settings.index'],
            ['ability' => 'admin.activities.access', 'route' => 'admin.activities.index'],
            ['ability' => 'admin.accessibility_control.access', 'route' => 'admin.accessibility-control.index'],
        ];

        foreach ($candidates as $candidate) {
            if ($user->hasAdminAbility($candidate['ability'])) {
                return route($candidate['route']);
            }
        }

        return route('admin.login');
    }
}

