<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\SettingUpdateRequest;
use App\Models\DeliveryServer;
use App\Models\CustomerGroup;
use App\Models\Setting;
use App\Models\TranslationLocale;
use App\Services\DeliveryServerService;
use App\Services\SettingService;
use App\Services\UpdateServerService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Str;

class SettingController extends Controller
{
    public function __construct(
        protected SettingService $settingService,
        protected UpdateServerService $updateServerService
    ) {}

    private function billingKeysToHide(): array
    {
        return [
            'billing_provider',
            'billing_default_provider',
            'billing_providers',

            'stripe_public_key',
            'stripe_secret',
            'stripe_webhook_secret',

            'paypal_client_id',
            'paypal_client_secret',

            'razorpay_key_id',
            'razorpay_key_secret',
            'razorpay_webhook_secret',

            'paystack_public_key',
            'paystack_secret',

            'flutterwave_public_key',
            'flutterwave_secret',
            'flutterwave_encryption_key',
            'flutterwave_webhook_secret',

            'manual_bank_name',
            'manual_account_name',
            'manual_account_number',
            'manual_instructions',
            'manual_qr_image_path',
        ];
    }

    private function normalizePurchaseCode(?string $code): string
    {
        $code = is_string($code) ? $code : '';
        $code = trim($code);

        $code = str_replace([
            "\u{2010}",
            "\u{2011}",
            "\u{2012}",
            "\u{2013}",
            "\u{2014}",
            "\u{2015}",
            "\u{2212}",
            "\u{FE63}",
            "\u{FF0D}",
        ], '-', $code);

        $code = preg_replace('/[\s\x{00A0}\x{200B}\x{200C}\x{200D}\x{FEFF}]+/u', '', $code) ?? '';

        if ($code !== '' && preg_match('/[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}/i', $code, $m)) {
            return $m[0];
        }

        $code = preg_replace('/[^0-9a-f\-]/i', '', $code) ?? '';

        if ($code !== '' && preg_match('/^[0-9a-f]{32}$/i', $code)) {
            return substr($code, 0, 8)
                . '-' . substr($code, 8, 4)
                . '-' . substr($code, 12, 4)
                . '-' . substr($code, 16, 4)
                . '-' . substr($code, 20, 12);
        }

        if ($code !== '' && preg_match('/[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}/i', $code, $m2)) {
            return $m2[0];
        }

        return $code;
    }

    /**
     * Display the settings page.
     */
    public function index(Request $request)
    {
        Setting::firstOrCreate(
            ['key' => 'site_language'],
            [
                'category' => 'general',
                'value' => 'en',
                'type' => 'string',
                'description' => 'Site language used as default locale.',
                'is_public' => false,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'brand_color'],
            [
                'category' => 'appearance',
                'value' => '#3b82f6',
                'type' => 'string',
                'description' => 'Primary brand color used across the public website (buttons, links, borders).',
                'is_public' => true,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'public_meta_image'],
            [
                'category' => 'appearance',
                'value' => null,
                'type' => 'string',
                'description' => 'Default meta/OG image used for social sharing (Open Graph / Twitter) on the public website.',
                'is_public' => true,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'cron_web_enabled'],
            [
                'category' => 'cron',
                'value' => 0,
                'type' => 'boolean',
                'description' => 'Enable/disable running the Laravel scheduler via the Web Cron URL.',
                'is_public' => false,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'cron_web_token'],
            [
                'category' => 'cron',
                'value' => Str::random(40),
                'type' => 'string',
                'description' => 'Security token for the Web Cron URL. Regenerate if it leaks.',
                'is_public' => false,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'cron_last_run_at'],
            [
                'category' => 'cron',
                'value' => null,
                'type' => 'string',
                'description' => 'Last time the scheduler was triggered by Web Cron or Run Now.',
                'is_public' => false,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'home_page_variant'],
            [
                'category' => 'appearance',
                'value' => '1',
                'type' => 'string',
                'description' => 'Select which homepage design to use for the public website.',
                'is_public' => true,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'public_header_sticky'],
            [
                'category' => 'appearance',
                'value' => 0,
                'type' => 'boolean',
                'description' => 'Make the public header sticky (fixed to top while scrolling).',
                'is_public' => true,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'home_redirect_enabled'],
            [
                'category' => 'appearance',
                'value' => 0,
                'type' => 'boolean',
                'description' => 'When enabled, visitors to the home page (/) will be redirected to the URL below.',
                'is_public' => true,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'home_redirect_url'],
            [
                'category' => 'appearance',
                'value' => '',
                'type' => 'string',
                'description' => 'Redirect destination. Use a full URL (https://...) or a path starting with /.',
                'is_public' => true,
            ]
        );

        foreach (['1', '2', '3', '4'] as $homeVariant) {
            Setting::firstOrCreate(
                ['key' => 'home_' . $homeVariant . '_text_map'],
                [
                    'category' => 'homepage',
                    'value' => json_encode([]),
                    'type' => 'json',
                    'description' => 'JSON map of {"original text": "replacement"} for Home ' . $homeVariant . '. Exact match required.',
                    'is_public' => true,
                ]
            );
        }

        $navDefaults = [
            'nav_show_features' => 1,
            'nav_show_pricing' => 1,
            'nav_show_blog' => 1,
            'nav_show_roadmap' => 1,
            'nav_show_docs' => 1,
            'nav_show_api_docs' => 1,
        ];

        foreach ($navDefaults as $key => $defaultValue) {
            Setting::firstOrCreate(
                ['key' => $key],
                [
                    'category' => 'navigation',
                    'value' => $defaultValue,
                    'type' => 'boolean',
                    'description' => 'Show/hide this item in the public navbar.',
                    'is_public' => true,
                ]
            );
        }

        Setting::firstOrCreate(
            ['key' => 'google_analytics_tracking_id'],
            [
                'category' => 'general',
                'value' => null,
                'type' => 'string',
                'description' => 'Google Analytics Measurement ID (e.g. G-XXXXXXXXXX) or UA Tracking ID (e.g. UA-XXXXXXXX-X).',
                'is_public' => true,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'transactional_delivery_server_id'],
            [
                'category' => 'email',
                'value' => null,
                'type' => 'string',
                'description' => 'Default delivery server for system transactional emails (verification/password reset). Use "system" to use MAIL_* from .env.',
                'is_public' => false,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'verification_delivery_server_id'],
            [
                'category' => 'email',
                'value' => null,
                'type' => 'string',
                'description' => 'Delivery server for verification emails. Use "inherit" to use the transactional default, or "system" to use MAIL_* from .env.',
                'is_public' => false,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'password_reset_delivery_server_id'],
            [
                'category' => 'email',
                'value' => null,
                'type' => 'string',
                'description' => 'Delivery server for password reset emails. Use "inherit" to use the transactional default, or "system" to use MAIL_* from .env.',
                'is_public' => false,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'gdpr_notice_title'],
            [
                'category' => 'privacy',
                'value' => 'We value your privacy',
                'type' => 'string',
                'description' => 'GDPR notice title shown on the public website.',
                'is_public' => true,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'gdpr_notice_description'],
            [
                'category' => 'privacy',
                'value' => 'We use cookies and similar technologies to improve your experience. You can accept or decline.',
                'type' => 'string',
                'description' => 'GDPR notice description shown on the public website.',
                'is_public' => true,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'gdpr_notice_accept_text'],
            [
                'category' => 'privacy',
                'value' => 'Accept',
                'type' => 'string',
                'description' => 'Text for the accept button.',
                'is_public' => true,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'gdpr_notice_decline_text'],
            [
                'category' => 'privacy',
                'value' => 'Decline',
                'type' => 'string',
                'description' => 'Text for the decline button.',
                'is_public' => true,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'gdpr_notice_position'],
            [
                'category' => 'privacy',
                'value' => 'bottom_full_width',
                'type' => 'string',
                'description' => 'Banner position on the public website.',
                'is_public' => true,
            ]
        );

        Setting::firstOrCreate(
            ['key' => 'gdpr_notice_delay_seconds'],
            [
                'category' => 'privacy',
                'value' => 0,
                'type' => 'integer',
                'description' => 'Delay (in seconds) before showing the GDPR notice.',
                'is_public' => true,
            ]
        );

        $category = $request->get('category', 'general');
        $settings = $this->settingService->getSettingsByCategory();
        $categories = $this->settingService->getCategories();

        $categories = array_values(array_filter($categories, fn ($cat) => $cat !== 'homepage'));

        if ($settings->has('billing')) {
            $keysToHide = $this->billingKeysToHide();
            $settings['billing'] = $settings['billing']->reject(fn ($s) => in_array((string) ($s->key ?? ''), $keysToHide, true));
        }

        if (!in_array($category, $categories, true)) {
            $category = $categories[0] ?? 'general';
        }

        $customerGroups = [];
        if ($category === 'account') {
            $customerGroups = CustomerGroup::orderBy('name')->pluck('name', 'id')->toArray();
        }

        $activeLocales = [];
        if ($category === 'general') {
            $activeLocales = TranslationLocale::query()
                ->where('is_active', true)
                ->orderBy('code')
                ->get(['code', 'name']);
        }

        $deliveryServerOptions = [];
        if ($category === 'email') {
            $deliveryServerOptions[''] = 'Auto';

            try {
                $deliveryServerService = app(DeliveryServerService::class);
                if ($deliveryServerService->hasSystemSmtpConfigured()) {
                    $deliveryServerOptions['system'] = 'System (SMTP)';
                }
            } catch (\Throwable $e) {
                // ignore
            }

            $activeServers = DeliveryServer::query()
                ->where('status', 'active')
                ->orderByDesc('is_primary')
                ->orderBy('name')
                ->get(['id', 'name', 'type']);

            foreach ($activeServers as $server) {
                $label = $server->name;
                if (is_string($server->type) && trim($server->type) !== '') {
                    $label .= ' (' . strtoupper(str_replace('-', ' ', $server->type)) . ')';
                }

                $deliveryServerOptions[(string) $server->id] = $label;
            }
        }

        $updateProduct = null;
        $updateChangelogs = null;
        $updateDownloadUrl = null;
        $updateConfig = null;
        $updateLicenseCheck = null;
        $updateStatus = null;
        $updateInstallState = null;
        $updateLastFailureAt = null;
        $updateLastFailureVersion = null;
        $updateLastFailureReason = null;

        $updateStatus = Cache::get('update_server:update_status');
        $updateInstallState = Setting::get('update_install_state');
        $updateLastFailureAt = Setting::get('update_last_failure_at');
        $updateLastFailureVersion = Setting::get('update_last_failure_version');
        $updateLastFailureReason = Setting::get('update_last_failure_reason');

        if (in_array($category, ['updates', 'changelogs'], true)) {
            $refresh = $request->boolean('refresh');
            $refreshUpdates = $category === 'updates' ? true : $refresh;
            $refreshChangelogs = $category === 'changelogs' ? true : $refresh;

            if ($category === 'updates' && $refreshUpdates) {
                try {
                    Artisan::call('updates:check', ['--force' => true]);
                } catch (\Throwable $e) {
                    // ignore
                }

                $updateStatus = Cache::get('update_server:update_status');
            }

            $baseUrl = (string) config('services.update_server.base_url', Setting::get('update_api_base_url'));
            $productId = config('services.update_server.product_id', Setting::get('update_product_id'));
            $productName = (string) config('services.update_server.product_name', Setting::get('update_product_name'));
            $productSecret = (string) config('services.update_server.product_secret', Setting::get('update_product_secret'));

            $appUrl = (string) config('app.url');
            $parsed = parse_url($appUrl);
            $domain = is_array($parsed) && is_string($parsed['host'] ?? null) && trim((string) $parsed['host']) !== ''
                ? (string) $parsed['host']
                : $appUrl;

            $licenseKey = Setting::get('update_license_key');
            $licenseKey = $this->normalizePurchaseCode(is_string($licenseKey) ? $licenseKey : null);

            if (is_string($licenseKey) && trim($licenseKey) !== '' && preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $licenseKey)) {
                $cacheKey = 'update_server:license_check:' . md5((string) $baseUrl) . ':' . md5((string) $licenseKey) . ':' . md5((string) $domain);
                if ($refresh) {
                    Cache::forget($cacheKey);
                }

                $updateLicenseCheck = Cache::remember($cacheKey, now()->addMinutes(10), function () use ($licenseKey, $domain, $productSecret, $productName) {
                    return $this->updateServerService->licenseCheck(
                        (string) $licenseKey,
                        $domain,
                        (string) $productSecret,
                        (string) $productName
                    );
                });
            }

            $updateConfig = [
                'base_url' => $baseUrl,
                'product_id' => $productId,
                'product_name' => $productName,
                'has_product_secret' => is_string($productSecret) && trim($productSecret) !== '',
            ];

            $productIdToUse = is_numeric($productId) ? (int) $productId : null;

            $baseUrlOk = is_string($baseUrl) && trim($baseUrl) !== '';
            $productIdOk = is_numeric($productIdToUse) && (int) $productIdToUse > 0;

            if ($baseUrlOk && $productIdOk) {
                if ($updateProduct === null) {
                    $updateProduct = $this->updateServerService->getProduct((int) $productIdToUse, (string) $baseUrl, $refreshUpdates);
                }

                if ($category !== 'changelogs' || ($updateChangelogs === null)) {
                    $updateChangelogs = $this->updateServerService->getChangelogs((int) $productIdToUse, (string) $baseUrl, $refreshChangelogs);
                }
            } else {
                $updateProduct = [
                    'success' => false,
                    'data' => null,
                    'message' => 'Update server settings are not configured yet.',
                    'status' => null,
                ];
                $updateChangelogs = $updateProduct;
            }

            $updateDownloadUrl = session()->pull('update_download_url');
        }

        return view('admin.settings.index', compact(
            'settings',
            'categories',
            'category',
            'customerGroups',
            'activeLocales',
            'deliveryServerOptions',
            'updateProduct',
            'updateChangelogs',
            'updateDownloadUrl',
            'updateConfig',
            'updateLicenseCheck',
            'updateStatus',
            'updateInstallState',
            'updateLastFailureAt',
            'updateLastFailureVersion',
            'updateLastFailureReason'
        ));
    }

    public function regenerateCronToken(Request $request)
    {
        Setting::set('cron_web_token', Str::random(40), 'cron', 'string');

        return redirect()
            ->route('admin.settings.index', ['category' => 'cron'])
            ->with('success', 'Cron token regenerated successfully. Update your cPanel cron job to use the new URL.');
    }

    public function runCronNow(Request $request)
    {
        $exitCode = 1;
        $output = '';

        try {
            $exitCode = Artisan::call('schedule:run');
            $output = (string) Artisan::output();
        } catch (\Throwable $e) {
            $output = $e->getMessage();
        }

        Setting::set('cron_last_run_at', now()->toIso8601String(), 'cron', 'string');

        return redirect()
            ->route('admin.settings.index', ['category' => 'cron'])
            ->with('success', 'Scheduler run finished (exit code: ' . (int) $exitCode . ').')
            ->with('cron_run_output', $output);
    }

    public function installUpdate(Request $request)
    {
        $targetVersion = $request->input('target_version');
        $targetVersion = is_string($targetVersion) ? trim($targetVersion) : '';

        if ($targetVersion === '') {
            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', 'Latest version is not available to install yet. Please refresh and try again.');
        }

        $installedVersion = trim((string) config('mailpurse.version', ''));
        if ($installedVersion === '') {
            $installedVersion = Setting::get('app_version');
            $installedVersion = is_string($installedVersion) ? trim($installedVersion) : '';
        }
        if ($installedVersion !== '' && version_compare($targetVersion, $installedVersion, '<=')) {
            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', 'No update available.');
        }

        $state = Setting::get('update_install_state');
        if (
            is_array($state)
            && (
                (bool) ($state['in_progress'] ?? false)
                || in_array((string) ($state['status'] ?? ''), ['queued', 'running'], true)
            )
        ) {
            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', 'An update is already running. Please wait.');
        }

        Setting::set('update_install_state', [
            'in_progress' => false,
            'status' => 'queued',
            'message' => 'Update queued.',
            'version' => $targetVersion,
            'queued_at' => now()->toIso8601String(),
        ], 'updates', 'json');

        return redirect()
            ->route('admin.settings.index', ['category' => 'updates'])
            ->with('success', 'Update queued. It will be installed automatically in background by cron. The site will enter maintenance mode during installation.');
    }

    public function requestUpdateDownload(Request $request)
    {
        $baseUrl = (string) config('services.update_server.base_url', Setting::get('update_api_base_url'));
        $licenseKey = $request->input('update_license_key');
        if (!is_string($licenseKey) || trim($licenseKey) === '') {
            $licenseKey = Setting::get('update_license_key');
        }
        $productSecret = (string) config('services.update_server.product_secret', Setting::get('update_product_secret'));
        $productName = (string) config('services.update_server.product_name', Setting::get('update_product_name'));

        $missing = [];

        if (!is_string($baseUrl) || trim($baseUrl) === '') {
            $missing[] = 'Update API Base URL';
        }
        if (!is_string($licenseKey) || trim($licenseKey) === '') {
            $missing[] = 'License Key';
        }
        if (!is_string($productSecret) || trim($productSecret) === '') {
            $missing[] = 'Product Secret';
        }
        if (!is_string($productName) || trim($productName) === '') {
            $missing[] = 'Product Name';
        }

        if (!empty($missing)) {
            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', 'Missing required update settings: ' . implode(', ', $missing) . '.');
        }

        $licenseKey = $this->normalizePurchaseCode(is_string($licenseKey) ? $licenseKey : null);
        if ($licenseKey === '' || !preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $licenseKey)) {
            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', 'Invalid purchase code format. Please use the Envato/ThemeForest purchase code (e.g. xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx).');
        }

        $appUrl = (string) config('app.url');
        $parsed = parse_url($appUrl);
        $domain = is_array($parsed) && is_string($parsed['host'] ?? null) && trim((string) $parsed['host']) !== ''
            ? (string) $parsed['host']
            : $appUrl;

        $licenseCheck = $this->updateServerService->licenseCheck(
            (string) $licenseKey,
            $domain,
            (string) $productSecret,
            (string) $productName
        );

        if (!($licenseCheck['valid'] ?? false)) {
            $licenseActivate = $this->updateServerService->licenseActivate(
                (string) $licenseKey,
                $domain,
                (string) $productSecret,
                (string) $productName
            );

            if (!($licenseActivate['valid'] ?? false)) {
                $message = is_string($licenseActivate['message'] ?? null) && trim((string) $licenseActivate['message']) !== ''
                    ? (string) $licenseActivate['message']
                    : (is_string($licenseCheck['message'] ?? null) && trim((string) $licenseCheck['message']) !== ''
                        ? (string) $licenseCheck['message']
                        : 'License key is not valid.');

                return redirect()
                    ->route('admin.settings.index', ['category' => 'updates'])
                    ->with('error', $message);
            }
        }

        $result = $this->updateServerService->requestDownloadUrl(
            (string) $baseUrl,
            (string) $licenseKey,
            $domain,
            (string) $productSecret,
            (string) $productName
        );

        if (!($result['success'] ?? false)) {
            $message = is_string($result['message'] ?? null) && trim((string) $result['message']) !== ''
                ? (string) $result['message']
                : 'Failed to get download link.';

            if (is_numeric($result['status'] ?? null)) {
                $message .= ' (HTTP ' . (int) $result['status'] . ')';
            }
            if (array_key_exists('raw', $result) && $result['raw'] !== null) {
                $raw = is_string($result['raw']) ? $result['raw'] : json_encode($result['raw']);
                if (is_string($raw) && trim($raw) !== '') {
                    $message .= ': ' . $raw;
                }
            }

            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', $message);
        }

        $downloadUrl = is_string($result['download_url'] ?? null) ? $result['download_url'] : null;

        if (!is_string($downloadUrl) || trim($downloadUrl) === '') {
            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', 'Download URL not available.');
        }

        return redirect()
            ->route('admin.settings.index', ['category' => 'updates'])
            ->with('success', 'Download link generated.')
            ->with('update_download_url', $downloadUrl);
    }

    public function activateLicense(Request $request)
    {
        $baseUrl = (string) config('services.update_server.base_url', Setting::get('update_api_base_url'));
        $licenseKey = $request->input('update_license_key');
        if (!is_string($licenseKey) || trim($licenseKey) === '') {
            $licenseKey = Setting::get('update_license_key');
        }
        $productSecret = (string) config('services.update_server.product_secret', Setting::get('update_product_secret'));
        $productName = (string) config('services.update_server.product_name', Setting::get('update_product_name'));

        $missing = [];

        if (!is_string($licenseKey) || trim($licenseKey) === '') {
            $missing[] = 'License Key';
        }
        if (!is_string($productSecret) || trim($productSecret) === '') {
            $missing[] = 'Product Secret';
        }

        if (!empty($missing)) {
            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', 'Missing required settings: ' . implode(', ', $missing) . '.');
        }

        $licenseKey = $this->normalizePurchaseCode(is_string($licenseKey) ? $licenseKey : null);
        if ($licenseKey === '' || !preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $licenseKey)) {
            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', 'Invalid purchase code format. Please use the Envato/ThemeForest purchase code (e.g. xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx).');
        }

        $appUrl = (string) config('app.url');
        $parsed = parse_url($appUrl);
        $domain = is_array($parsed) && is_string($parsed['host'] ?? null) && trim((string) $parsed['host']) !== ''
            ? (string) $parsed['host']
            : $appUrl;

        $result = $this->updateServerService->licenseActivate(
            (string) $licenseKey,
            $domain,
            (string) $productSecret,
            (string) $productName
        );

        $cacheKey = 'update_server:license_check:' . md5((string) $baseUrl) . ':' . md5((string) $licenseKey) . ':' . md5((string) $domain);
        Cache::forget($cacheKey);

        if (!($result['valid'] ?? false)) {
            $message = is_string($result['message'] ?? null) && trim((string) $result['message']) !== ''
                ? (string) $result['message']
                : 'Failed to activate license.';

            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', $message);
        }

        Setting::set('update_license_key', $licenseKey);

        $message = null;
        if (is_array($result['data'] ?? null) && is_string($result['data']['message'] ?? null)) {
            $message = trim((string) $result['data']['message']);
        }
        if (!is_string($message) || $message === '') {
            $message = 'License activated successfully.';
        }

        return redirect()
            ->route('admin.settings.index', ['category' => 'updates', 'refresh' => 1])
            ->with('success', $message);
    }

    public function deactivateLicense(Request $request)
    {
        $baseUrl = (string) config('services.update_server.base_url', Setting::get('update_api_base_url'));
        $licenseKey = $request->input('update_license_key');
        if (!is_string($licenseKey) || trim($licenseKey) === '') {
            $licenseKey = Setting::get('update_license_key');
        }
        $productSecret = (string) config('services.update_server.product_secret', Setting::get('update_product_secret'));
        $productName = (string) config('services.update_server.product_name', Setting::get('update_product_name'));

        $missing = [];

        if (!is_string($baseUrl) || trim($baseUrl) === '') {
            $missing[] = 'Update API Base URL';
        }
        if (!is_string($licenseKey) || trim($licenseKey) === '') {
            $missing[] = 'License Key';
        }
        if (!is_string($productSecret) || trim($productSecret) === '') {
            $missing[] = 'Product Secret';
        }
        if (!is_string($productName) || trim($productName) === '') {
            $missing[] = 'Product Name';
        }

        if (!empty($missing)) {
            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', 'Missing required update settings: ' . implode(', ', $missing) . '.');
        }

        $licenseKey = $this->normalizePurchaseCode(is_string($licenseKey) ? $licenseKey : null);
        if ($licenseKey === '' || !preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $licenseKey)) {
            return redirect()
                ->route('admin.settings.index', ['category' => 'updates'])
                ->with('error', 'Invalid purchase code format. Please use the Envato/ThemeForest purchase code (e.g. xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx).');
        }

        $appUrl = (string) config('app.url');
        $parsed = parse_url($appUrl);
        $domain = is_array($parsed) && is_string($parsed['host'] ?? null) && trim((string) $parsed['host']) !== ''
            ? (string) $parsed['host']
            : $appUrl;

        $result = $this->updateServerService->licenseDeactivate(
            (string) $licenseKey,
            $domain,
            (string) $productSecret,
            (string) $productName
        );

        $cacheKey = 'update_server:license_check:' . md5((string) $baseUrl) . ':' . md5((string) $licenseKey) . ':' . md5((string) $domain);
        Cache::forget($cacheKey);

        if (!($result['success'] ?? false) && !($result['valid'] ?? false)) {
            $data = $result['data'] ?? null;
            $status = is_array($data) ? ($data['status'] ?? null) : null;
            if (!is_string($status) || !in_array(strtolower(trim($status)), ['success', 'deactivated'], true)) {
                $message = is_string($result['message'] ?? null) && trim((string) $result['message']) !== ''
                    ? (string) $result['message']
                    : 'Failed to deactivate license.';

                return redirect()
                    ->route('admin.settings.index', ['category' => 'updates'])
                    ->with('error', $message);
            }
        }

        Setting::set('update_license_key', null);

        $message = null;
        if (is_array($result['data'] ?? null) && is_string($result['data']['message'] ?? null)) {
            $message = trim((string) $result['data']['message']);
        }
        if (!is_string($message) || $message === '') {
            $message = 'License deactivated.';
        }

        return redirect()
            ->route('admin.settings.index', ['category' => 'updates', 'refresh' => 1])
            ->with('success', $message);
    }

    /**
     * Update settings.
     */
    public function update(SettingUpdateRequest $request)
    {
        $category = $request->input('category', 'general');
        $categories = $this->settingService->getCategories();

        $categories = array_values(array_filter($categories, fn ($cat) => $cat !== 'homepage'));

        $brandingDisk = (string) config('filesystems.branding_disk', 'public');

        if (!in_array($category, $categories, true)) {
            $category = $categories[0] ?? 'general';
        }
        $data = $request->except(['_token', '_method', 'category']);

        if ($category === 'billing') {
            foreach ($this->billingKeysToHide() as $key) {
                unset($data[$key]);
            }
        }

        if ($request->hasFile('app_logo')) {
            $file = $request->file('app_logo');

            if ($file && $file->isValid()) {
                $oldPath = Setting::get('app_logo');
                if (is_string($oldPath) && $oldPath !== '') {
                    Storage::disk($brandingDisk)->delete($oldPath);
                }

                $path = $file->storePublicly('branding', $brandingDisk);
                $data['app_logo'] = $path;
            }
        }

        if ($request->hasFile('app_logo_dark')) {
            $file = $request->file('app_logo_dark');

            if ($file && $file->isValid()) {
                $oldPath = Setting::get('app_logo_dark');
                if (is_string($oldPath) && $oldPath !== '') {
                    Storage::disk($brandingDisk)->delete($oldPath);
                }

                $path = $file->storePublicly('branding', $brandingDisk);
                $data['app_logo_dark'] = $path;
            }
        }

        if ($request->hasFile('site_favicon')) {
            $file = $request->file('site_favicon');

            if ($file && $file->isValid()) {
                $oldPath = Setting::get('site_favicon');
                if (is_string($oldPath) && $oldPath !== '') {
                    Storage::disk($brandingDisk)->delete($oldPath);
                }

                $path = $file->storePublicly('branding', $brandingDisk);
                $data['site_favicon'] = $path;
            }
        }

        if ($request->hasFile('public_meta_image')) {
            $file = $request->file('public_meta_image');

            if ($file && $file->isValid()) {
                $oldPath = Setting::get('public_meta_image');
                if (is_string($oldPath) && $oldPath !== '') {
                    Storage::disk($brandingDisk)->delete($oldPath);
                }

                $path = $file->storePublicly('branding', $brandingDisk);
                $data['public_meta_image'] = $path;
            }
        }

        $booleanKeys = Setting::query()
            ->where('type', 'boolean')
            ->where('category', $category)
            ->pluck('key');

        foreach ($booleanKeys as $key) {
            if (!array_key_exists($key, $data)) {
                $data[$key] = 0;
            }
        }

        foreach (['google_client_secret', 's3_secret', 'wasabi_secret', 'update_product_secret', 'update_license_key', 'openai_api_key', 'gemini_api_key'] as $secretKey) {
            if (!array_key_exists($secretKey, $data)) {
                continue;
            }

            $secret = $data[$secretKey];
            if (!is_string($secret) || trim($secret) === '' || trim($secret) === '********') {
                unset($data[$secretKey]);
            }
        }

        if ($category === 'updates' && array_key_exists('update_license_key', $data)) {
            $licenseKey = $data['update_license_key'];

            if (is_string($licenseKey) && trim($licenseKey) !== '') {
                $licenseKey = $this->normalizePurchaseCode($licenseKey);
                if ($licenseKey === '' || !preg_match('/^[0-9a-f]{8}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{4}-[0-9a-f]{12}$/i', $licenseKey)) {
                    return redirect()
                        ->route('admin.settings.index', ['category' => $category])
                        ->with('error', 'Invalid purchase code format. Please use the Envato/ThemeForest purchase code (e.g. xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx).');
                }

                $data['update_license_key'] = $licenseKey;
            }
        }

        if (array_key_exists('site_language', $data)) {
            Cache::forget('translation_locales:site');
        }

        $this->settingService->updateSettings($data);

        return redirect()
            ->route('admin.settings.index', ['category' => $category])
            ->with('success', 'Settings updated successfully.');
    }

    public function revealSecret(Request $request, string $key)
    {
        $allowed = [
            'google_client_secret',
            's3_secret',
            'wasabi_secret',
            'update_product_secret',
            'update_license_key',
            'openai_api_key',
            'gemini_api_key',
        ];

        if (!in_array($key, $allowed, true)) {
            abort(404);
        }

        $value = Setting::get($key);
        $value = is_string($value) ? trim($value) : '';

        return response()->json([
            'success' => true,
            'value' => $value,
        ]);
    }
}

