<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\DeliveryServer;
use App\Models\Setting;
use Illuminate\Http\Request;

class IntegrationController extends Controller
{
    public function index(Request $request)
    {
        $tab = (string) $request->query('tab', 'delivery-servers');
        if (!in_array($tab, ['delivery-servers', 'google', 'wordpress'], true)) {
            $tab = 'delivery-servers';
        }

        $providers = [
            [
                'key' => 'mailgun',
                'label' => 'Mailgun',
                'type' => 'mailgun',
                'description' => 'Send emails using Mailgun API and webhooks.',
                'supported' => true,
            ],
            [
                'key' => 'sendgrid',
                'label' => 'SendGrid',
                'type' => 'sendgrid',
                'description' => 'Send emails with SendGrid API and event webhooks.',
                'supported' => true,
            ],
            [
                'key' => 'postmark',
                'label' => 'Postmark',
                'type' => 'postmark',
                'description' => 'Deliver transactional emails through Postmark.',
                'supported' => true,
            ],
            [
                'key' => 'sparkpost',
                'label' => 'SparkPost',
                'type' => 'sparkpost',
                'description' => 'Deliver emails using SparkPost API.',
                'supported' => true,
            ],
            [
                'key' => 'amazon-ses',
                'label' => 'Amazon SES',
                'type' => 'amazon-ses',
                'description' => 'Send emails through Amazon SES (API).',
                'supported' => true,
            ],
            [
                'key' => 'zeptomail-api',
                'label' => 'ZeptoMail API',
                'type' => 'zeptomail-api',
                'description' => 'Send emails via ZeptoMail Email API (templates or raw HTML/text).',
                'supported' => true,
            ],
        ];

        $supportedTypes = collect($providers)
            ->filter(fn ($p) => (bool) ($p['supported'] ?? false))
            ->pluck('type')
            ->values()
            ->all();

        $configured = DeliveryServer::query()
            ->whereIn('type', $supportedTypes)
            ->whereNull('customer_id')
            ->selectRaw('type, COUNT(*) as cnt')
            ->groupBy('type')
            ->pluck('cnt', 'type')
            ->map(fn ($v) => ((int) $v) > 0)
            ->toArray();

        $serversByType = DeliveryServer::query()
            ->whereIn('type', $supportedTypes)
            ->whereNull('customer_id')
            ->orderByDesc('id')
            ->get()
            ->groupBy('type')
            ->map(fn ($items) => $items->first())
            ->all();

        $googleSocialiteAvailable = class_exists('Laravel\\Socialite\\Facades\\Socialite');

        $clientId = config('services.google.client_id') ?: env('GOOGLE_CLIENT_ID');
        $clientSecret = config('services.google.client_secret') ?: env('GOOGLE_CLIENT_SECRET');

        $dbClientId = Setting::get('google_client_id');
        $dbClientSecret = Setting::get('google_client_secret');

        if (is_string($dbClientId)) {
            $dbClientId = trim($dbClientId);
        }
        if (is_string($dbClientSecret)) {
            $dbClientSecret = trim($dbClientSecret);
        }

        $googleClientId = (is_string($dbClientId) && $dbClientId !== '') ? $dbClientId : $clientId;
        $googleClientSecret = (is_string($dbClientSecret) && $dbClientSecret !== '') ? $dbClientSecret : $clientSecret;

        $googleOAuthConfigured = (bool) ($googleSocialiteAvailable && $googleClientId && $googleClientSecret);

        $googleRedirectSheets = route('customer.integrations.google.callback', ['service' => 'sheets']);
        $googleRedirectDrive = route('customer.integrations.google.callback', ['service' => 'drive']);

        return view('admin.integrations.index', compact(
            'tab',
            'providers',
            'configured',
            'serversByType',
            'googleSocialiteAvailable',
            'googleOAuthConfigured',
            'googleClientId',
            'googleClientSecret',
            'googleRedirectSheets',
            'googleRedirectDrive'
        ));
    }

    public function downloadWordpressPlugin(Request $request)
    {
        $path = base_path('wordpress-plugin/mailpurse-integration.zip');

        abort_unless(is_file($path), 404);

        return response()->download($path, 'mailpurse-integration.zip');
    }
}
